import { CarOfferConfigSchema } from 'entities/Car/types/CarOfferConfigSchema';
import { CarOfferSchema } from 'entities/Car/types/CarOfferSchema';
import { GeoSuggestSchema } from 'entities/Geo/types/GeoSuggestSchema';

function extractItemTitle({ id, title }: { id: string; title: string }): { id: string; title: string } {
    return { id, title };
}

function extractItemName({ id, name }: { id: string; name: string }): { id: string; title: string } {
    return { id, title: name };
}

function extractLocation(data: { lon: number; lat: number; location_name: string; id: string }): GeoSuggestSchema {
    return {
        lon: data.lon,
        lat: data.lat,
        name: data.location_name,
        geoid: Number(data.id),
    };
}

const DEFAULT_CURRENCY = [
    { id: 'id_eur', title: 'EUR' },
    { id: 'id_czk', title: 'CZK' },
];

const DEFAULT_STATUSES = [
    { id: 'draft', title: 'Draft' },
    { id: 'confirmed', title: 'Confirmed' },
    { id: 'paid', title: 'Paid' },
];

const FIXES_CURRENCY = {
    CZK: 'id_czk',
    EUR: 'id_eur',
};

function currenciesFix(list: Array<{ id: string; title: string }>): Array<{ id: string; title: string }> {
    return list.map((item) => {
        return {
            id: FIXES_CURRENCY.hasOwnProperty(item.id) ? FIXES_CURRENCY[item.id] : item.id,
            title: item.title,
        };
    });
}

export function parseCarOfferConfig(offer: CarOfferSchema): CarOfferConfigSchema {
    let currencies: CarOfferConfigSchema['currencies'] = DEFAULT_CURRENCY;
    let statuses: CarOfferConfigSchema['statuses'] = DEFAULT_STATUSES;
    let insurance_types: CarOfferConfigSchema['insurance_types'];
    let delivery_locations: CarOfferConfigSchema['delivery_locations'] = [];
    let return_locations: CarOfferConfigSchema['return_locations'] = [];
    let limit_km_per_day: CarOfferConfigSchema['limit_km_per_day'];
    let overrun_cost_per_km: CarOfferConfigSchema['overrun_cost_per_km'];

    for (const item of offer.primary) {
        switch (item.id) {
            case 'currencies':
                currencies = currenciesFix(item.currencies.map(extractItemName));
                break;
            case 'statuses':
                statuses = item.statuses.map(extractItemTitle);
                break;
            case 'insurance_types':
                insurance_types = item.types.map(extractItemTitle);
                break;
            case 'delivery_locations':
                delivery_locations = item.locations.map(extractLocation);
                break;
            case 'return_locations':
                return_locations = item.locations.map(extractLocation);
                break;
            case 'limit_km_per_day':
                limit_km_per_day = item.default_value;
                break;
            case 'overrun_cost_per_km':
                overrun_cost_per_km = item.default_value;
                break;
        }
    }

    return {
        currencies,
        statuses,
        insurance_types,
        delivery_locations,
        return_locations,
        limit_km_per_day,
        overrun_cost_per_km,
        offer_options: offer.secondary.map(extractItemTitle),
    };
}
