import * as React from 'react';

import { fetchCarsSearch } from 'entities/Car/api/fetchCarsSearch/fetchCarsSearch';
import { getCarModelName } from 'entities/Car/helpers/getCarModelName/getCarModelName';
import { getCarNumber } from 'entities/Car/helpers/getCarNumber/getCarNumber';
import { CarSearchSchema } from 'entities/Car/types/CarSearchSchema';

import { Suggest, SuggestProps, SuggestValueState } from 'shared/ui/Suggest/Suggest';

import styles from 'entities/Car/ui/CarInput/CarInput.css';

export interface CarInputProps
    extends Omit<
        SuggestProps<CarSearchSchema>,
        'initialValue' | 'onSuggestChange' | 'dataProvider' | 'menuItemsProvider'
    > {
    initialValue?: CarSearchSchema;

    onCarInputChange?(car: Optional<CarSearchSchema>): void;
}

function carInputText(item: CarSearchSchema): string {
    const { number, vin } = item;

    const carNumber = getCarNumber(number, vin);

    return getCarModelName(item) + ' ' + carNumber;
}

// @todo: add storybook test
export const CarInput: React.FC<CarInputProps> = function CarInput({ initialValue, onCarInputChange, ...otherProps }) {
    const menuItemsProvider = React.useCallback((items: CarSearchSchema[]) => {
        return items.map((item) => {
            const { id, number, vin } = item;

            const carNumber = getCarNumber(number, vin);

            return {
                id,
                text: carInputText(item),
                data: item,
                content: (
                    <span className={styles.menu}>
                        <span className={styles.model}>{getCarModelName(item)}</span>

                        {carNumber}
                    </span>
                ),
            };
        });
    }, []);

    const onChange = React.useCallback(
        (textValue: string, dataValue: Optional<CarSearchSchema>) => {
            if (onCarInputChange) {
                onCarInputChange(dataValue);
            }
        },
        [onCarInputChange],
    );

    const suggestInitValue = React.useMemo<Optional<SuggestValueState<CarSearchSchema>>>(() => {
        if (!initialValue) {
            return undefined;
        }

        return {
            id: initialValue.id,
            text: carInputText(initialValue),
            data: initialValue,
        };
    }, [initialValue]);

    return (
        <Suggest<CarSearchSchema>
            {...otherProps}
            initialValue={suggestInitValue}
            onSuggestChange={onChange}
            dataProvider={fetchCarsSearch}
            menuItemsProvider={menuItemsProvider}
        />
    );
};
