import * as React from 'react';

import { getHardwareFilterIdsFlag } from 'utils/getHardwareFilterIdsFlag';

import { CarsFilter } from 'entities/Car/consts/CarsFilter';

import { CheckboxItemOptions } from 'shared/ui/Checkbox/Checkbox';
import { CheckboxGroup, CheckboxGroupProps } from 'shared/ui/CheckboxGroup/CheckboxGroup';
import { FilterGroupProps } from 'shared/ui/FilterGroup/FilterGroup';

import { i18n } from 'entities/Car/ui/CarsHardwareFilter/CarsHardwareFilter.i18n';

export interface CarsHardwareFilterProps extends Pick<FilterGroupProps, 'title'>, Pick<CheckboxGroupProps, 'disabled'> {
    className?: string;
    values?: string[];

    onChangeFilter(value: Record<string, boolean>): void;
}

export const CarsHardwareFilter: React.FC<CarsHardwareFilterProps> = function CarsHardwareFilter({
    className,
    values,
    onChangeFilter,
    ...otherProps
}) {
    const allowedIds = getHardwareFilterIdsFlag();

    const onChangeHandler = React.useCallback(
        (values: string[]) => {
            onChangeFilter({
                [CarsFilter.CAMERA]: Boolean(values && values.includes(CarsFilter.CAMERA)),
                [CarsFilter.TELEMATICS]: Boolean(values && values.includes(CarsFilter.TELEMATICS)),
            });
        },
        [onChangeFilter],
    );

    const items = React.useMemo<CheckboxItemOptions[]>(() => {
        let items = [
            {
                id: CarsFilter.CAMERA,
                label: i18n('With camera'),
            },

            {
                id: CarsFilter.TELEMATICS,
                label: i18n('With telematics'),
            },
        ];

        if (allowedIds) {
            items = items.filter(({ id }) => allowedIds.includes(id));
        }

        return items;
    }, [allowedIds]);

    return (
        <CheckboxGroup
            {...otherProps}
            values={values}
            items={items}
            title={i18n('Hardware')}
            onChange={onChangeHandler}
        />
    );
};
