import * as React from 'react';

import { useCarParksList, UseCarParksListResource } from 'entities/Car/api/useCarParksList/useCarParksList';
import { CarsFilter } from 'entities/Car/consts/CarsFilter';
import { CarParksSchema } from 'entities/Car/types/CarParksSchema';

import { CheckboxItemOptions } from 'shared/ui/Checkbox/Checkbox';
import { CheckboxGroup, CheckboxGroupProps } from 'shared/ui/CheckboxGroup/CheckboxGroup';
import { FilterGroup, FilterGroupProps } from 'shared/ui/FilterGroup/FilterGroup';
import { Spin } from 'shared/ui/Spin/Spin';

import { i18n } from 'entities/Car/ui/CarsParksFilter/CarsParksFilter.i18n';

import styles from 'entities/Car/ui/CarsParksFilter/CarsParksFilter.css';

export interface CarsParksFilterProps extends Pick<FilterGroupProps, 'title'>, Pick<CheckboxGroupProps, 'disabled'> {
    className?: string;
    values: Nullable<string>;

    onChangeFilter(type: string, value: string): void;
}

const FILTER_GROUP_TITLE = i18n('Taxi Parks');

function getCheckboxItems(items: Nullable<CarParksSchema[]> = []): Optional<CheckboxItemOptions[]> {
    return items?.map(({ name, park_id }) => {
        return {
            label: name,
            id: park_id,
        };
    });
}

const CarsParksFilterInternal: React.FC<CarsParksFilterProps & { parksResource: UseCarParksListResource }> =
    function CarsParksFilterInternal({ className, parksResource, values, onChangeFilter, ...otherProps }) {
        const parks = parksResource.read();

        const items = React.useMemo(() => getCheckboxItems(parks), [parks]);

        const onChangeHandler = React.useCallback(
            (values: string[]) => {
                onChangeFilter(CarsFilter.PARK_ID, values.join(','));
            },
            [onChangeFilter],
        );

        if (!items) {
            return null;
        }

        return (
            <CheckboxGroup
                {...otherProps}
                className={className}
                title={FILTER_GROUP_TITLE}
                items={items}
                values={values?.split(',')}
                onChange={onChangeHandler}
            />
        );
    };

export const CarsParksFilter: React.FC<CarsParksFilterProps> = function CarsParksFilter({ className, ...otherProps }) {
    const parksResource = useCarParksList();

    const fallback = (
        <FilterGroup
            className={className}
            title={FILTER_GROUP_TITLE}
        >
            <Spin className={styles.loader} />
        </FilterGroup>
    );

    return (
        <React.Suspense fallback={fallback}>
            <CarsParksFilterInternal
                className={className}
                parksResource={parksResource}
                {...otherProps}
            />
        </React.Suspense>
    );
};
