import * as React from 'react';
import cn from 'classnames/bind';

import { getCarNumber } from 'entities/Car/helpers/getCarNumber/getCarNumber';
import { CarOfferPreviewSchema } from 'entities/Car/types/CarOfferPreviewSchema';
import { CarSearchSchema } from 'entities/Car/types/CarSearchSchema';

import { Path } from 'shared/consts/Path';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { Link } from 'shared/ui/Link/Link';
import { Shimmer } from 'shared/ui/Shimmer/Shimmer';
import { Spin } from 'shared/ui/Spin/Spin';
import { Widget } from 'shared/ui/Widget/Widget';

import { i18n } from 'entities/Car/ui/CarsScheduleList/CarsScheduleList.i18n';

import styles from 'entities/Car/ui/CarsScheduleList/CarsScheduleList.css';

export interface CarsScheduleListItems {
    cars?: Array<{
        id: string;
        car_details: CarSearchSchema;
        offers?: CarOfferPreviewSchema[];
    }>;

    isFullyLoaded?: boolean;
}

export interface CarsScheduleListProps {
    className?: string;
    items: CarsScheduleListItems;
    hoveredItem?: string;
    isLoading?: boolean;

    onMouseEnterItem?(event: React.MouseEvent<HTMLLIElement>): void;

    onMouseLeaveItem?(): void;

    fetchNextPage?(): void;
}

const cx = cn.bind(styles);

export const CarsScheduleList: React.FC<CarsScheduleListProps> = function CarsScheduleList({
    className,
    items,
    hoveredItem,
    isLoading,
    onMouseEnterItem,
    onMouseLeaveItem,
    fetchNextPage,
}) {
    const { cars, isFullyLoaded } = items;

    const itemLoading = (
        <li className={cx(styles.item, { empty: true, center: true })}>
            <Spin />
        </li>
    );

    const content = cars?.length ? (
        <>
            {cars.map(({ id, car_details }) => {
                return (
                    <li
                        className={cx(styles.item, { hovered: id === hoveredItem })}
                        onMouseEnter={onMouseEnterItem}
                        onMouseLeave={onMouseLeaveItem}
                        data-id={id}
                        key={id}
                    >
                        <Link
                            className={styles.model}
                            href={generateRouterPath(Path.CAR, { id })}
                        >
                            {car_details.model_name}
                        </Link>

                        <span className={styles.number}>{getCarNumber(car_details.number, car_details.vin, true)}</span>
                    </li>
                );
            })}

            {isLoading && itemLoading}

            {!isFullyLoaded && !isLoading && (
                <li className={cx(styles.item, { empty: true })}>
                    <Link onClick={fetchNextPage}>{i18n('Show more')}</Link>
                </li>
            )}
        </>
    ) : (
        <li className={cx(styles.item, { empty: true, center: true })}>{i18n('No information')}</li>
    );

    return (
        <Widget
            className={cn(styles.content, className)}
            contentContainer
        >
            <div className={styles.header}>
                {!cars && isLoading ? (
                    <Shimmer className={styles.shimmer} />
                ) : (
                    i18n('{count} car', {
                        // @todo: get correct value
                        count: cars?.length || 0,
                    })
                )}

                {/* @todo: temporary disabled */}
                <Link
                    className={styles.filter}
                    disabled
                >
                    {i18n('Show filters')}
                </Link>
            </div>

            <ul className={styles.list}>{!cars && isLoading ? itemLoading : content}</ul>
        </Widget>
    );
};
