import { getI18nLocale } from 'utils/language/getI18nLocale';

import { GeoSuggestSchema } from 'entities/Geo/types/GeoSuggestSchema';

import { I18nLocale } from 'shared/consts/I18nLocale';
import { buildFetchRequestResult, fetchRequest } from 'shared/helpers/fetchRequest/fetchRequest';
import { CacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';

export type FetchGeoSuggestLang = 'ru_RU' | 'en_US' | 'cs_CZ';

// Docs: https://wiki.yandex-team.ru/suggest/geo/params/
export interface FetchGeoSuggestData extends FetchGeoSuggestOptions {
    v: number;
    search_type: string;
    outformat: 'json';
    origin: 'maps';
}

export interface FetchGeoSuggestOptions {
    bases: string;
    countries?: string;
    part: string;
    results: number;
    lang: FetchGeoSuggestLang;
}

export interface FetchGeoSuggestServerRes {
    part: string;
    results: GeoSuggestSchema[];
}

export type FetchGeoSuggestRes = FetchGeoSuggestServerRes['results'];

const LANG_MAPPING: Record<I18nLocale, FetchGeoSuggestLang> = {
    'ru-RU': 'ru_RU',
    'en-US': 'en_US',
    'cs-CZ': 'en_US', // cs_CZ have some names with ru_RU fallbacks
};

export function fetchGeoSuggest(
    text: string,
    options: Partial<FetchGeoSuggestOptions> = {},
    cache?: CacheRequestContext,
) {
    if (!text) {
        return buildFetchRequestResult(Promise.resolve([]));
    }

    return fetchRequest<FetchGeoSuggestData, FetchGeoSuggestRes, FetchGeoSuggestServerRes>(
        '/suggest-geo',
        {
            v: 7,
            outformat: 'json',
            origin: 'maps',
            lang: LANG_MAPPING[getI18nLocale()] || 'en_US',
            search_type: 'taxi',
            bases: 'geo,biz',
            part: text,
            results: 10,
            ...options,
        },
        {
            cache,
            parser(res) {
                return res.results;
            },
        },
    );
}
