import * as React from 'react';

import { fetchGeoSuggest, FetchGeoSuggestOptions } from 'entities/Geo/api/fetchGeoSuggest/fetchGeoSuggest';
import { GeoSuggestSchema } from 'entities/Geo/types/GeoSuggestSchema';

import { buildFetchRequestResult } from 'shared/helpers/fetchRequest/fetchRequest';
import { CacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { Suggest, SuggestProps, SuggestValueState } from 'shared/ui/Suggest/Suggest';

import PinIcon from 'shared/ui/Icons/images/pin-16.inline.svg';

export interface GeoInputData {
    geoid?: number;
    name: string;
    location: [number, number];
}

export interface GeoInputProps
    extends Omit<
        SuggestProps<GeoSuggestSchema>,
        'initialValue' | 'onSuggestChange' | 'dataProvider' | 'menuItemsProvider' | 'hasArrow' | 'icon'
    > {
    initialValue?: GeoInputData;

    geoSuggestOptions?: Partial<FetchGeoSuggestOptions>;
    defaultItems?: GeoSuggestSchema[];

    onGeoInputChange?(carId: Optional<GeoInputData>): void;
}

// @todo: add storybook test
export const GeoInput: React.FC<GeoInputProps> = function GeoInput({
    onGeoInputChange,
    initialValue,
    defaultItems,
    geoSuggestOptions,
    ...otherProps
}) {
    const menuItemsProvider = React.useCallback((items: GeoSuggestSchema[]) => {
        return items.map((item, ind) => {
            const text = [item.name, item.desc].filter(Boolean).join(', ');

            return {
                id: String(item.geoid) + '-' + ind,
                text,
                data: item,
            };
        });
    }, []);

    const onChange = React.useCallback((textValue: string, dataValue: Optional<GeoSuggestSchema>) => {
        if (onGeoInputChange) {
            if (dataValue) {
                onGeoInputChange({
                    geoid: dataValue.geoid,
                    name: [dataValue.name, dataValue.desc].filter(Boolean).join(', '),
                    location: [dataValue.lon, dataValue.lat],
                });
            } else {
                onGeoInputChange(undefined);
            }
        }
    }, []);

    const dataProvider = React.useCallback(
        (text: string, cacheContext: CacheRequestContext) => {
            if (defaultItems && !text) {
                return buildFetchRequestResult(Promise.resolve(defaultItems));
            }

            return fetchGeoSuggest(text, geoSuggestOptions, cacheContext);
        },
        [geoSuggestOptions, defaultItems],
    );

    const suggestInitValue = React.useMemo<Optional<SuggestValueState<GeoSuggestSchema>>>(() => {
        if (!initialValue) {
            return undefined;
        }

        const geoId = initialValue.geoid || -1;

        return {
            id: String(geoId),
            text: initialValue.name,
            data: {
                geoid: geoId,
                name: initialValue.name,
                lon: initialValue.location && initialValue.location[0],
                lat: initialValue.location && initialValue.location[1],
            },
        };
    }, [initialValue]);

    return (
        <Suggest<GeoSuggestSchema>
            {...otherProps}
            icon={PinIcon}
            hasArrow={false}
            onSuggestChange={onChange}
            initialValue={suggestInitValue}
            dataProvider={dataProvider}
            menuItemsProvider={menuItemsProvider}
        />
    );
};
