import { ONE_DAY, ONE_SECOND } from 'constants/constants';

import { formatCarModel } from 'entities/Car';
import { CarStatusType } from 'entities/Car/consts/CarStatusType';
import { CarStatusSchema } from 'entities/Car/types/CarStatusSchema';

import { DateAggregation } from 'shared/consts/DateAggregation';
import { fixCurrency } from 'shared/helpers/fixCurrency/fixCurrency';
import { getArrayTop } from 'shared/helpers/getArrayTop/getArrayTop';
import { sortCarTop } from 'shared/helpers/sortCarTop/sortCarTop';
import { useFetchRequest } from 'shared/hooks/useFetchRequest/useFetchRequest';
import { ChartPoint } from 'shared/types/ChartPoint';

const RIDES_SUMMARY_TIMEOUT = 10000000; // 10 seconds

export interface RideSummaryByTimeValue {
    bookings: number;
    fleet_utilization: number;
    rental_utilization: number;
    revenue: number;
    total_engine_hours: number;
    total_mileage: number;
    average_check: number;
    total_duration_hours: number;
}

export interface RideSummaryByTime {
    timestamp: number;
    value: RideSummaryByTimeValue;
}

export interface RideSummaryCarInfoValues {
    duration_hours: number;
    average_check: number;
    revenue: number;
    fleet_utilization: number;
    rental_utilization: number;
    riding_duration_hours: number;
    mileage: number;
}

export interface RideSummaryByCar {
    model_id: string;
    name: string;
    number: string;
    object_id: string;
    values: RideSummaryCarInfoValues;
}

export interface UseRidesSummaryServerRes {
    car_count: number;
    addon_sales: {
        child_seat: number;
        entry_to_eco_zones_in_germany: number;
        gps: number;
        roof_rack: number;
        snow_chains: number;
    };

    result_by_car: RideSummaryByCar[];
    gistogram_value: RideSummaryByTime[];
    insurance_sales: Record<string, number>[];

    statuses: CarStatusSchema[];
    total_values: {
        rentals: number;
        rentals_avg_per_time: number;
        rental_utilization_total_for_the_period: number;

        bookings_avg_per_time: number;
        bookings_total_for_the_period: number;

        currency: 'CZK' | 'EUR';

        engine_hours_avg_per_time: number;
        engine_hours_total_for_the_period: number;

        fleet_utilization_total_for_the_period: number;
        utilization_total_for_the_period: number; // will be deprecated

        mileage_avg_per_time: number;
        mileage_total_for_the_period: number;

        revenue_avg_per_time: number;
        revenue_total_for_the_period: number;

        average_check: number;
    };
}

export interface FetchRidesSummaryData {
    since?: Nullable<number>;
    until?: Nullable<number>;
    aggregation: DateAggregation;
    tariff?: UseRidesSummaryTariff;
}

export enum UseRidesSummaryTariff {
    STANDART_OFFER = 'standart_offer',
    RENTAL_OFFER = 'rental_offer',
}

export interface UseRidesSummaryReq {
    timeout: number;
    since?: number;
    until?: number;
    limit?: number; // default: 5000
    aggregation?: DateAggregation;
    tariff?: UseRidesSummaryTariff;
    car_id?: string;
}

export interface UseRidesSummaryInsuranceSale {
    title: string;
    id: string;
    count: number;
}

export interface UseRidesSummaryAddonSale {
    title: string;
    id: string;
    count: number;
}

export interface CarTop {
    model_id: string;
    name: string;
    number: string;
    object_id: string;
    value: number;
}

type RidesSummaryStatuses = Record<CarStatusType, number>;

export interface UseRidesSummaryRes {
    topMileageCars: CarTop[];
    worseMileageCars: CarTop[];

    topHoursCars: CarTop[];
    worseHoursCars: CarTop[];

    topUtilizationCars: CarTop[];
    worseUtilizationCars: CarTop[];

    topRentalUtilizationCars: CarTop[];
    worseRentalUtilizationCars: CarTop[];

    topRevenueCars: CarTop[];
    worseRevenueCars: CarTop[];

    topAverageCheckCars: CarTop[];
    worseAverageCheckCars: CarTop[];

    mileageChartPoints: ChartPoint[];
    hoursChartPoints: ChartPoint[];
    utilizationChartPoints: ChartPoint[];
    rentalUtilizationChartPoints: ChartPoint[];
    averageCheckChartPoints: ChartPoint[];
    bookingsChartPoints: ChartPoint[];
    revenueChartPoints: ChartPoint[];

    averageDurationByTime: number;
    averageMileageByTime: number;
    averageFleetUtilization: number;
    averageRentalUtilization: number;
    totalDuration: number;
    totalMileage: number;

    rentals: number;
    bookingsAvgPerTime: number;
    bookingsTotalForThePeriod: number;
    rentalsAvgPerTime: number;
    revenueAvgPerTime: number;
    revenueTotalForThePeriod: number;
    averageCheck: number;

    currency: string;

    totalCars: number;

    statuses?: RidesSummaryStatuses;

    addonSales: UseRidesSummaryAddonSale[];
    insuranceSales: UseRidesSummaryInsuranceSale[];
}

interface CarsSummary {
    mileageCars: CarTop[];
    hoursCars: CarTop[];
    fleetUtilizationCars: CarTop[];
    rentalUtilizationCars: CarTop[];
    averageCheck: CarTop[];
    revenue: CarTop[];
}

const TOP_COUNT = 5;
const PRECISION = 10000;

function getChartPoints(
    ridesSummaryByTime: RideSummaryByTime[],
    key: keyof RideSummaryByTimeValue,
    isPercent: boolean = false,
): ChartPoint[] {
    return ridesSummaryByTime.map((ride) => {
        let value = ride.value[key];
        const formattedValue = isPercent ? Math.round(value * PRECISION) / 100 : value;
        const date = ride.timestamp * ONE_SECOND;

        return {
            begin_date: date,
            end_date: date + ONE_DAY - 1,
            line1: formattedValue,
        };
    });
}

function ridesSummaryParser(res: UseRidesSummaryServerRes): UseRidesSummaryRes {
    const { result_by_car, gistogram_value, statuses, total_values, car_count, addon_sales, insurance_sales } = res;
    const totalCars = car_count || result_by_car.length;
    const carsSummary: CarsSummary = result_by_car.reduce(
        (acc, car) => {
            acc.mileageCars.push({
                object_id: car.object_id,
                name: car.name,
                model_id: car.model_id,
                number: car.number,
                value: car.values.mileage,
            });
            acc.hoursCars.push({
                object_id: car.object_id,
                name: car.name,
                model_id: car.model_id,
                number: car.number,
                value: car.values.riding_duration_hours,
            });
            acc.fleetUtilizationCars.push({
                object_id: car.object_id,
                name: car.name,
                model_id: car.model_id,
                number: car.number,
                value: car.values.fleet_utilization,
            });
            acc.rentalUtilizationCars.push({
                object_id: car.object_id,
                name: car.name,
                model_id: car.model_id,
                number: car.number,
                value: car.values.rental_utilization,
            });
            acc.averageCheck.push({
                object_id: car.object_id,
                name: car.name,
                model_id: car.model_id,
                number: car.number,
                value: car.values.average_check,
            });
            acc.revenue.push({
                object_id: car.object_id,
                name: car.name,
                model_id: car.model_id,
                number: car.number,
                value: car.values.revenue,
            });

            return acc;
        },
        {
            mileageCars: [],
            hoursCars: [],
            fleetUtilizationCars: [],
            rentalUtilizationCars: [],
            averageCheck: [],
            revenue: [],
        } as CarsSummary,
    );

    sortCarTop(carsSummary.mileageCars);
    sortCarTop(carsSummary.hoursCars);
    sortCarTop(carsSummary.fleetUtilizationCars);
    sortCarTop(carsSummary.rentalUtilizationCars);
    sortCarTop(carsSummary.averageCheck);
    sortCarTop(carsSummary.revenue);

    return {
        topMileageCars: getArrayTop(carsSummary.mileageCars, TOP_COUNT, true),
        worseMileageCars: getArrayTop(carsSummary.mileageCars, TOP_COUNT),

        topHoursCars: getArrayTop(carsSummary.hoursCars, TOP_COUNT, true),
        worseHoursCars: getArrayTop(carsSummary.hoursCars, TOP_COUNT),

        topUtilizationCars: getArrayTop(carsSummary.fleetUtilizationCars, TOP_COUNT, true),
        worseUtilizationCars: getArrayTop(carsSummary.fleetUtilizationCars, TOP_COUNT),

        topRentalUtilizationCars: getArrayTop(carsSummary.rentalUtilizationCars, TOP_COUNT, true),
        worseRentalUtilizationCars: getArrayTop(carsSummary.rentalUtilizationCars, TOP_COUNT),

        topRevenueCars: getArrayTop(carsSummary.revenue, TOP_COUNT, true),
        worseRevenueCars: getArrayTop(carsSummary.revenue, TOP_COUNT),

        topAverageCheckCars: getArrayTop(carsSummary.averageCheck, TOP_COUNT, true),
        worseAverageCheckCars: getArrayTop(carsSummary.averageCheck, TOP_COUNT),

        mileageChartPoints: getChartPoints(gistogram_value, 'total_mileage'),
        hoursChartPoints: getChartPoints(gistogram_value, 'total_engine_hours', false),
        utilizationChartPoints: getChartPoints(gistogram_value, 'fleet_utilization', true),
        rentalUtilizationChartPoints: getChartPoints(gistogram_value, 'rental_utilization', true),
        averageCheckChartPoints: getChartPoints(gistogram_value, 'average_check'),

        bookingsChartPoints: getChartPoints(gistogram_value, 'bookings'),
        revenueChartPoints: getChartPoints(gistogram_value, 'revenue'),

        averageDurationByTime: total_values.engine_hours_avg_per_time,
        averageMileageByTime: total_values.mileage_avg_per_time,
        averageFleetUtilization: total_values.fleet_utilization_total_for_the_period,
        averageRentalUtilization: total_values.rental_utilization_total_for_the_period,

        totalDuration: total_values.engine_hours_total_for_the_period,
        totalMileage: total_values.mileage_total_for_the_period,

        rentals: total_values.rentals,
        bookingsAvgPerTime: total_values.bookings_avg_per_time,
        bookingsTotalForThePeriod: total_values.bookings_total_for_the_period,
        rentalsAvgPerTime: total_values.rentals_avg_per_time || 0,
        revenueAvgPerTime: total_values.revenue_avg_per_time,
        revenueTotalForThePeriod: total_values.revenue_total_for_the_period,
        averageCheck: total_values.average_check,

        currency: fixCurrency(total_values.currency),

        totalCars,

        statuses: statuses?.reduce((memo, status) => {
            memo[status.filter_id] = status.cars_count;

            return memo;
        }, {} as RidesSummaryStatuses),

        addonSales: Object.keys(addon_sales).map((key) => {
            return {
                id: key,
                title: formatCarModel(key),
                count: addon_sales[key],
            };
        }),

        insuranceSales: insurance_sales.map((insurance) => {
            const keys = Object.keys(insurance);
            const id = keys[0];

            return {
                id: id,
                title: id,
                count: insurance[id],
            };
        }),
    };
}

export function useRidesSummary(data: FetchRidesSummaryData) {
    let req: UseRidesSummaryReq = {
        timeout: RIDES_SUMMARY_TIMEOUT,
        aggregation: data.aggregation,
    };

    if (data.tariff) {
        req.tariff = data.tariff;
    }

    if (data.since && data.until) {
        req.since = data.since;
        req.until = data.until;
    }

    return useFetchRequest<UseRidesSummaryReq, UseRidesSummaryRes, UseRidesSummaryServerRes>(
        '/api/leasing/user/orders_summary',
        req,
        {
            parser: ridesSummaryParser,
            timeout: 10000,
        },
    );
}
