import { EMPTY_DATA } from 'constants/constants';

import { isRidesBookingColumnsFlag } from 'utils/isRidesBookingColumnsFlag';

import { RideSchema } from 'entities/Ride/types/RideSchema';

import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';

import { i18n } from 'entities/Ride/helpers/getRideBookingDetails/getRideBookingDetails.i18n';

export interface GetRideBookingDetailsResult {
    delivery_location: string;
    return_location: string;
    mileage_start: string;
    mileage_finish: string;
    fuel_tank_level_start: string;
    fuel_tank_level_finish: string;
    booking_total: string;
    booking_deposit: string;
    offer_options: string[];
    insurance_type: string;
}

function formatMileage(num: string | undefined): string {
    if (typeof num === 'string') {
        const value = formatNumber(Number(num), {
            minimumFractionDigits: 0,
            maximumFractionDigits: 0,
        });

        return i18n('{value} km', { value });
    }

    return EMPTY_DATA;
}

export function getRideBookingDetails(ride: RideSchema): Optional<GetRideBookingDetailsResult> {
    if (!isRidesBookingColumnsFlag()) {
        return undefined;
    }

    const segment = ride.segment;
    const bill = segment?.bill_dm || [];

    const billIndex = bill.reduce((memo: Record<string, any>, item) => {
        memo[item.type] = item.value;

        return memo;
    }, {});

    return {
        delivery_location: segment?.delivery_location || EMPTY_DATA,
        return_location: segment?.return_location || EMPTY_DATA,
        mileage_start: formatMileage(billIndex.mileage_start),
        mileage_finish: formatMileage(billIndex.mileage_finish),
        fuel_tank_level_start: billIndex.fuel_tank_level_start ? billIndex.fuel_tank_level_start + '%' : EMPTY_DATA,
        fuel_tank_level_finish: billIndex.fuel_tank_level_finish ? billIndex.fuel_tank_level_finish + '%' : EMPTY_DATA,
        booking_total: billIndex.total || EMPTY_DATA,
        booking_deposit: billIndex.deposit || EMPTY_DATA,
        insurance_type: billIndex.insurance_type || EMPTY_DATA,
        offer_options: billIndex.offer_options || [],
    };
}
