import React from 'react';

import { ALL_SIGNALS_UNCHECKED_KEY, SIGNALS_PAGE_SIZE } from 'constants/constants';

import { isSignalsSupportFlag } from 'utils/isSignalsSupportFlag';

import { formatServerSignals } from 'entities/Signal/helpers/formatServerSignals/formatServerSignals';
import { SignalDescriptionDictSchema } from 'entities/Signal/types/SignalDescriptionDictSchema';
import { SignalFormattedSchema } from 'entities/Signal/types/SignalFormattedSchema';
import { SignalSchema } from 'entities/Signal/types/SignalSchema';

import { useFetchInfinite } from 'shared/hooks/useFetchInifinite/useFetchInfinite';

const SIGNAL_LIST_TIMEOUT = 10000;

export interface UseSignalListServerData {
    since?: Nullable<number>;
    until?: Nullable<number>;
    actual_only?: Nullable<boolean>;
    source?: Nullable<string>;
    signals?: Nullable<string>;
    cars_ids?: Nullable<string>;
    vins?: Nullable<string>;
    models?: Nullable<string>;
    numbers?: Nullable<string>;
    // 'car_in_deny_zone',
    // 'leasing_car_is_blocked',
    // 'telematics_tag_alerting',
    // 'scoring_trace_tag',
    page_size?: number;
    cars_cursor?: Nullable<number>;
    sessions_cursor?: Nullable<number>;
    is_support_mode?: boolean;
}

interface UseSignalListServerRes {
    can_get_more_pages: boolean;

    next_cars_cursor?: number;
    next_sessions_cursor?: number;

    signals: SignalSchema[];
    signals_descriptions: SignalDescriptionDictSchema;
}

interface UseSignalListData extends Omit<UseSignalListServerData, 'vins' | 'models' | 'numbers' | 'page_number'> {
    car?: Nullable<string>;
}

export type UseSignalListRes = SignalFormattedSchema[];

function mergeNextPage<
    T extends { cars_cursor?: number; sessions_cursor?: number },
    S extends { can_get_more_pages?: boolean; next_cars_cursor?: number; next_sessions_cursor?: number },
>(prevNext: Optional<T>, resData: S): Optional<T> {
    if (prevNext && resData.can_get_more_pages) {
        const nextPage = { ...prevNext };

        if (resData.next_sessions_cursor) {
            nextPage.sessions_cursor = resData.next_sessions_cursor;
        }
        if (resData.next_cars_cursor) {
            nextPage.cars_cursor = resData.next_cars_cursor;
        }

        return nextPage;
    }

    return undefined;
}

export function useSignalList(data: UseSignalListData) {
    const options = React.useMemo(() => {
        return {
            timeout: SIGNAL_LIST_TIMEOUT,
            mergeNext: mergeNextPage,
            mergeData(
                prevData: Optional<UseSignalListRes>,
                resData: UseSignalListServerRes,
            ): Optional<UseSignalListRes> {
                const data = formatServerSignals(resData.signals, resData.signals_descriptions);

                if (!prevData) {
                    return data;
                }

                return [...prevData, ...data];
            },
        };
    }, []);

    const carFilter = data.car
        ? {
              numbers: data.car.replace(/\s+/g, ''),
              car: undefined,
          }
        : {};

    let payload: UseSignalListServerData = {
        ...data,
        ...carFilter,
        page_size: data.page_size || SIGNALS_PAGE_SIZE,
    };

    if (data.signals === ALL_SIGNALS_UNCHECKED_KEY) {
        delete payload.signals;
    }

    if (isSignalsSupportFlag()) {
        payload.is_support_mode = true;
        payload.signals = 'signalq_support_preapproved_tag';
    }

    return useFetchInfinite<UseSignalListServerData, UseSignalListRes, UseSignalListServerRes>(
        '/api/leasing/signals/list',
        payload,
        options,
    );
}
