import * as React from 'react';

import { getCarSignalDisplayName } from 'entities/Car/helpers/getCarSignalDisplayName/getCarSignalDisplayName';
import {
    useSignalsDescriptions,
    UseSignalsDescriptionsResource,
} from 'entities/Signal/api/useSignalsDescriptions/useSignalsDescriptions';
import { SignalsFilter } from 'entities/Signal/consts/SignalsFilter';
import { SignalDescriptionSchema } from 'entities/Signal/types/SignalDescriptionSchema';
import { SignalSchema } from 'entities/Signal/types/SignalSchema';

import { CheckboxItemOptions } from 'shared/ui/Checkbox/Checkbox';
import { CheckboxGroup, CheckboxGroupProps } from 'shared/ui/CheckboxGroup/CheckboxGroup';
import { FilterGroup, FilterGroupProps } from 'shared/ui/FilterGroup/FilterGroup';
import { Spin } from 'shared/ui/Spin/Spin';

import { i18n } from 'entities/Signal/ui/SignalsDescriptionsFilter/SignalsDescriptionsFilter.i18n';

import styles from 'entities/Signal/ui/SignalsDescriptionsFilter/SignalsDescriptionsFilter.css';

export interface SignalsDescriptionsFilterProps
    extends Pick<FilterGroupProps, 'title'>,
        Pick<CheckboxGroupProps, 'disabled'> {
    className?: string;
    values: Nullable<string>;

    onChangeFilter(type: string, value: string): void;
}

const DEFAULT_VALUE = 'all';

const FILTER_GROUP_TITLE = i18n('Signals');

function getCheckboxItems(items: SignalDescriptionSchema[] = []): CheckboxItemOptions[] {
    return items.map((item) => {
        const { display_name, name } = item;

        return {
            // @todo: backend return only russian language :( https://nda.ya.ru/t/drXr4org4pBq5r
            label: getCarSignalDisplayName({} as SignalSchema, item) || display_name,
            id: name,
        };
    });
}

// @todo: add storybook test
const SignalsDescriptionsFilterInternal: React.FC<
    SignalsDescriptionsFilterProps & { resource: UseSignalsDescriptionsResource }
> = function SignalsDescriptionsFilterInternal({ className, resource, values, onChangeFilter, ...otherProps }) {
    const signals = resource.read();

    const items = React.useMemo(() => getCheckboxItems(signals), [signals]);

    const onChangeHandler = React.useCallback(
        (values: string[]) => {
            onChangeFilter(SignalsFilter.SIGNALS, values.join(',') || DEFAULT_VALUE);
        },
        [onChangeFilter],
    );

    return (
        <CheckboxGroup
            {...otherProps}
            className={className}
            title={FILTER_GROUP_TITLE}
            items={items}
            values={values?.split(',')}
            onChange={onChangeHandler}
        />
    );
};

export const SignalsDescriptionsFilter: React.FC<SignalsDescriptionsFilterProps> = function SignalsDescriptionsFilter({
    className,
    ...otherProps
}) {
    const resource = useSignalsDescriptions();

    const fallback = (
        <FilterGroup
            className={className}
            title={FILTER_GROUP_TITLE}
        >
            <Spin className={styles.loader} />
        </FilterGroup>
    );

    return (
        <React.Suspense fallback={fallback}>
            <SignalsDescriptionsFilterInternal
                className={className}
                resource={resource}
                {...otherProps}
            />
        </React.Suspense>
    );
};
