import { ONE_MILLISECOND, ONE_SECOND } from 'constants/constants';

import { UserBaseSchema } from 'entities/User/types/UserBaseSchema';
import { UserDrivingLicenseFormSchema } from 'entities/User/types/UserDrivingLicenseFormSchema';
import { UserPassportFormSchema } from 'entities/User/types/UserPassportFormSchema';
import { UserPhotoContentTypeSchema, UserPhotosSchema } from 'entities/User/types/UserPhotosSchema';
import { UserSchema } from 'entities/User/types/UserSchema';
import { UserSetupSchema } from 'entities/User/types/UserSetupSchema';

import { useFetch, UseFetchResource } from 'shared/hooks/useFetch/useFetch';

export interface UseUserReq {
    user_id: string;
}

export type UseUserServerRes = UserSchema;

export interface UseUserRes extends UserBaseSchema {
    address: string;
    photos: UseUserPhotosRes;
    documents: {
        passport?: UserPassportFormSchema;
        driving_license?: UserDrivingLicenseFormSchema;
    };
    passport_revision: Optional<Date>;
    driving_license_revision: Optional<Date>;
    setup: UserSetupSchema;
}

export type UseUserDocumentPhotoRes = Partial<
    Record<UserPhotoContentTypeSchema, { userId?: string; photoId?: string; date?: Date }>
>;

export interface UseUserPhotosRes {
    passport: UseUserDocumentPhotoRes;
    driving_license: UseUserDocumentPhotoRes;
    submitted_at?: Date;
}

export type UseUserResource = UseFetchResource<UseUserRes>;

function getDocumentPhotos(userId: string, data: UserPhotosSchema): UseUserPhotosRes {
    let res: UseUserPhotosRes = {
        passport: {},
        driving_license: {},
    };
    let submittedAt;

    for (let key in data) {
        const document = data[key];

        let photos = {} as UseUserDocumentPhotoRes;

        for (let key of Object.keys(document)) {
            document[key].map(({ id: photoId, type, submitted_at }) => {
                photos[type] = {
                    userId,
                    photoId,
                    date: new Date(submitted_at * ONE_SECOND),
                };

                if (!submittedAt || submittedAt > submitted_at) {
                    submittedAt = submitted_at;
                }
            });
        }

        res[key] = photos;
    }

    return { ...res, submitted_at: submittedAt ? new Date(submittedAt * ONE_SECOND) : undefined };
}

function formatStringArray(value: Optional<string>): Optional<string | string[]> {
    if (!value) {
        return undefined;
    }

    const arr = value.split(',');

    return arr.length > 1 ? arr : arr[0];
}

function getRevisionDate(rev: Optional<string>): Optional<Date> {
    const timestamp = rev?.match(/\d+$/);

    return timestamp ? new Date(Number(timestamp) / ONE_MILLISECOND) : undefined;
}

function getDate(str: Optional<string>): Optional<Date> {
    return str ? new Date(str) : undefined;
}

function userParser({
    id,
    status,
    first_name,
    last_name,
    pn,
    address,
    photos,
    documents,
    passport_revision,
    driving_license_revision,
    setup,
}: UseUserServerRes): UseUserRes {
    const passport = documents?.passport && documents?.passport[0];
    const driving_license = documents?.driving_license && documents?.driving_license[0];

    return {
        id,
        status,
        first_name,
        last_name,
        pn,
        address: address || '',
        photos: getDocumentPhotos(id, photos),
        documents: {
            passport: passport
                ? {
                      ...passport,
                      birth_date: getDate(passport?.birth_date),
                      issue_date: getDate(passport?.issue_date),
                  }
                : undefined,
            driving_license: driving_license
                ? {
                      ...driving_license,
                      birth_date: getDate(driving_license?.birth_date),
                      categories: formatStringArray(driving_license?.categories),
                      issue_date: getDate(driving_license?.issue_date),
                      categories_b_valid_from_date: getDate(driving_license?.categories_b_valid_from_date),
                      categories_b_valid_to_date: getDate(driving_license?.categories_b_valid_to_date),
                  }
                : undefined,
        },
        passport_revision: getRevisionDate(passport_revision),
        driving_license_revision: getRevisionDate(driving_license_revision),
        setup,
    };
}

export function useUser(id: string) {
    return useFetch<UseUserReq, UseUserRes, UseUserServerRes>(
        '/api/leasing/user/get-info',
        {
            user_id: id,
        },
        { parser: userParser },
    );
}
