import * as React from 'react';

import { getUserDocumentDriverLicenseField } from 'entities/User/index';
import { UserDrivingLicenseFormSchema } from 'entities/User/types/UserDrivingLicenseFormSchema';

import { ButtonSize } from 'shared/consts/ButtonSize';
import { InputSize } from 'shared/consts/InputSize';
import { useForm } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { InputDateSingle } from 'shared/ui/InputDateSingle/InputDateSingle';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { Select } from 'shared/ui/Select/Select';

import { i18n } from 'entities/User/ui/UserDocumentDriverLicenseForm/UserDocumentDriverLicenseForm.i18n';

import styles from 'entities/User/ui/UserDocumentDriverLicenseForm/UserDocumentDriverLicenseForm.css';

export interface UserDocumentDriverLicenseFormProps {
    className?: string;

    initial?: UserDrivingLicenseFormSchema;
    controller?: UseFormControllerSub<UserDrivingLicenseFormSchema>;

    onFormChange?(state: UserDrivingLicenseFormSchema): void;
}

const INIT_FORM: OptionalRecord<UserDrivingLicenseFormSchema> = {
    first_name: '',
    last_name: '',
    front_country: '',
    categories: '',
    number_front: '',
    number_back: '',

    // unused fields
    middle_name: '',
    prev_licence_number: '',
    back_country: '',
};

const DATE_FORMAT: Intl.DateTimeFormatOptions = {
    day: '2-digit',
    month: 'short',
    year: 'numeric',
};
const CALENDAR_BACK_YEAR_STEP = 100;
const CALENDAR_NEXT_YEAR_STEP = 10;

export const UserDocumentDriverLicenseForm: React.FC<UserDocumentDriverLicenseFormProps> =
    function UserDocumentDriverLicenseForm({ className, initial, controller, onFormChange }) {
        const { update, watch, errors } = useForm<OptionalRecord<UserDrivingLicenseFormSchema>>({
            init: { ...INIT_FORM, ...initial },
            validation: {},
            controller,
            onFormChange,
        });

        const categoryList: MenuItemOptions[] = React.useMemo(
            () => [
                { label: 'A', value: 'a' },
                { label: 'B', value: 'b' },
            ],
            [],
        );

        return (
            <FormContainer className={className}>
                <div className={styles.half}>
                    <FormLabel title={getUserDocumentDriverLicenseField('first_name')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.first_name} />}>
                            <Input
                                inputSize={InputSize.M}
                                value={watch('first_name')}
                                hasError={Boolean(errors.first_name)}
                                hasClear
                                withBackground
                                onInputChange={update('first_name')}
                            />
                        </FormErrorContainer>
                    </FormLabel>

                    <FormLabel title={getUserDocumentDriverLicenseField('last_name')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.last_name} />}>
                            <Input
                                inputSize={InputSize.M}
                                value={watch('last_name')}
                                hasError={Boolean(errors.last_name)}
                                hasClear
                                withBackground
                                onInputChange={update('last_name')}
                            />
                        </FormErrorContainer>
                    </FormLabel>
                </div>

                <div className={styles.half}>
                    <FormLabel title={getUserDocumentDriverLicenseField('front_country')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.front_country} />}>
                            <Input
                                inputSize={InputSize.M}
                                value={watch('front_country')}
                                hasError={Boolean(errors.front_country)}
                                hasClear
                                withBackground
                                onInputChange={update('front_country')}
                            />
                        </FormErrorContainer>
                    </FormLabel>

                    <FormLabel title={getUserDocumentDriverLicenseField('birth_date')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.birth_date} />}>
                            <InputDateSingle
                                inputSize={InputSize.M}
                                dateFormat={DATE_FORMAT}
                                date={watch('birth_date')}
                                backYearStep={CALENDAR_BACK_YEAR_STEP}
                                hasError={Boolean(errors.birth_date)}
                                withBackground
                                onDateChange={update('birth_date')}
                            />
                        </FormErrorContainer>
                    </FormLabel>
                </div>

                <div className={styles.half}>
                    <FormLabel title={getUserDocumentDriverLicenseField('categories')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.categories} />}>
                            <Select
                                size={ButtonSize.M}
                                items={categoryList}
                                placeholder={i18n('Select')}
                                multiple
                                checkedMenu={watch('categories')}
                                hasError={Boolean(errors.categories)}
                                withBackground
                                onSelectChange={update('categories')}
                            />
                        </FormErrorContainer>
                    </FormLabel>

                    <FormLabel title={getUserDocumentDriverLicenseField('issue_date')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.issue_date} />}>
                            <InputDateSingle
                                inputSize={InputSize.M}
                                dateFormat={DATE_FORMAT}
                                date={watch('issue_date')}
                                backYearStep={CALENDAR_BACK_YEAR_STEP}
                                hasError={Boolean(errors.issue_date)}
                                withBackground
                                onDateChange={update('issue_date')}
                            />
                        </FormErrorContainer>
                    </FormLabel>
                </div>

                <div className={styles.half}>
                    <FormLabel title={getUserDocumentDriverLicenseField('categories_b_valid_from_date')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.categories_b_valid_from_date} />}>
                            <InputDateSingle
                                inputSize={InputSize.M}
                                dateFormat={DATE_FORMAT}
                                date={watch('categories_b_valid_from_date')}
                                backYearStep={CALENDAR_BACK_YEAR_STEP}
                                hasError={Boolean(errors.categories_b_valid_from_date)}
                                withBackground
                                onDateChange={update('categories_b_valid_from_date')}
                            />
                        </FormErrorContainer>
                    </FormLabel>

                    <FormLabel title={getUserDocumentDriverLicenseField('categories_b_valid_to_date')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.categories_b_valid_to_date} />}>
                            <InputDateSingle
                                inputSize={InputSize.M}
                                dateFormat={DATE_FORMAT}
                                date={watch('categories_b_valid_to_date')}
                                nextYearStep={CALENDAR_NEXT_YEAR_STEP}
                                hasError={Boolean(errors.categories_b_valid_to_date)}
                                withBackground
                                onDateChange={update('categories_b_valid_to_date')}
                            />
                        </FormErrorContainer>
                    </FormLabel>
                </div>

                <div className={styles.half}>
                    <FormLabel title={getUserDocumentDriverLicenseField('number_front')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.number_front} />}>
                            <Input
                                inputSize={InputSize.M}
                                value={watch('number_front')}
                                hasError={Boolean(errors.number_front)}
                                hasClear
                                withBackground
                                onInputChange={update('number_front')}
                            />
                        </FormErrorContainer>
                    </FormLabel>

                    <FormLabel title={getUserDocumentDriverLicenseField('number_back')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.number_back} />}>
                            <Input
                                inputSize={InputSize.M}
                                value={watch('number_back')}
                                hasError={Boolean(errors.number_back)}
                                hasClear
                                withBackground
                                onInputChange={update('number_back')}
                            />
                        </FormErrorContainer>
                    </FormLabel>
                </div>

                <ErrorMessage error={errors._serverError} />
            </FormContainer>
        );
    };
