import * as React from 'react';

import { getUserDocumentPassportField, getUserDocumentPassportGender } from 'entities/User/index';
import { UserPassportFormSchema } from 'entities/User/types/UserPassportFormSchema';

import { ButtonSize } from 'shared/consts/ButtonSize';
import { InputSize } from 'shared/consts/InputSize';
import { useForm } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { InputDateSingle } from 'shared/ui/InputDateSingle/InputDateSingle';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { Select } from 'shared/ui/Select/Select';

import { i18n } from 'entities/User/ui/UserDocumentPassportForm/UserDocumentPassportForm.i18n';

import styles from 'entities/User/ui/UserDocumentPassportForm/UserDocumentPassportForm.css';

export interface UserDocumentPassportFormProps {
    className?: string;

    initial?: UserPassportFormSchema;
    controller?: UseFormControllerSub<UserPassportFormSchema>;

    onFormChange?(state: UserPassportFormSchema): void;
}

const INIT_FORM: OptionalRecord<UserPassportFormSchema> = {
    first_name: '',
    last_name: '',
    gender: '',
    citizenship: '',
    doc_value: '',

    // unused fields
    middle_name: '',
    birth_place: '',
    subdivision_code: '',
    registration_apartment: '',
    registration_housing: '',
    registration_house: '',
    registration_letter: '',
    registration_street: '',
    registration_locality: '',
    registration_area: '',
    registration_region: '',
    biographical_country: '',
    registration_country: '',
};

const DATE_FORMAT: Intl.DateTimeFormatOptions = {
    day: '2-digit',
    month: 'short',
    year: 'numeric',
};
const CALENDAR_BACK_YEAR_STEP = 100;

export const UserDocumentPassportForm: React.FC<UserDocumentPassportFormProps> = function UserDocumentPassportForm({
    className,
    initial,
    controller,
    onFormChange,
}) {
    const { update, watch, errors } = useForm<OptionalRecord<UserPassportFormSchema>>({
        init: { ...INIT_FORM, ...initial },
        validation: {},
        controller,
        onFormChange,
    });

    const genderList: MenuItemOptions[] = React.useMemo(
        () => [
            { label: getUserDocumentPassportGender('male'), value: 'male' },
            { label: getUserDocumentPassportGender('female'), value: 'female' },
        ],
        [],
    );

    return (
        <FormContainer className={className}>
            <div className={styles.half}>
                <FormLabel title={getUserDocumentPassportField('first_name')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.first_name} />}>
                        <Input
                            inputSize={InputSize.M}
                            value={watch('first_name')}
                            hasError={Boolean(errors.first_name)}
                            hasClear
                            withBackground
                            onInputChange={update('first_name')}
                        />
                    </FormErrorContainer>
                </FormLabel>

                <FormLabel title={getUserDocumentPassportField('last_name')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.last_name} />}>
                        <Input
                            inputSize={InputSize.M}
                            value={watch('last_name')}
                            hasError={Boolean(errors.last_name)}
                            hasClear
                            withBackground
                            onInputChange={update('last_name')}
                        />
                    </FormErrorContainer>
                </FormLabel>
            </div>

            <div className={styles.half}>
                <FormLabel title={getUserDocumentPassportField('gender')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.gender} />}>
                        <Select
                            size={ButtonSize.M}
                            items={genderList}
                            placeholder={i18n('Select')}
                            checkedMenu={watch('gender')}
                            hasError={Boolean(errors.gender)}
                            withBackground
                            onSelectChange={update('gender')}
                        />
                    </FormErrorContainer>
                </FormLabel>

                <FormLabel title={getUserDocumentPassportField('birth_date')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.birth_date} />}>
                        <InputDateSingle
                            inputSize={InputSize.M}
                            dateFormat={DATE_FORMAT}
                            date={watch('birth_date')}
                            backYearStep={CALENDAR_BACK_YEAR_STEP}
                            hasError={Boolean(errors.birth_date)}
                            withBackground
                            onDateChange={update('birth_date')}
                        />
                    </FormErrorContainer>
                </FormLabel>
            </div>

            <FormLabel title={getUserDocumentPassportField('citizenship')}>
                <FormErrorContainer error={<ErrorLabel errors={errors.citizenship} />}>
                    <Input
                        inputSize={InputSize.M}
                        value={watch('citizenship')}
                        hasError={Boolean(errors.citizenship)}
                        hasClear
                        withBackground
                        onInputChange={update('citizenship')}
                    />
                </FormErrorContainer>
            </FormLabel>

            <div className={styles.half}>
                <FormLabel title={getUserDocumentPassportField('doc_value')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.doc_value} />}>
                        <Input
                            inputSize={InputSize.M}
                            value={watch('doc_value')}
                            hasError={Boolean(errors.doc_value)}
                            hasClear
                            withBackground
                            onInputChange={update('doc_value')}
                        />
                    </FormErrorContainer>
                </FormLabel>

                <FormLabel title={getUserDocumentPassportField('issue_date')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.issue_date} />}>
                        <InputDateSingle
                            inputSize={InputSize.M}
                            dateFormat={DATE_FORMAT}
                            date={watch('issue_date')}
                            backYearStep={CALENDAR_BACK_YEAR_STEP}
                            hasError={Boolean(errors.issue_date)}
                            withBackground
                            onDateChange={update('issue_date')}
                        />
                    </FormErrorContainer>
                </FormLabel>
            </div>

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
