import * as React from 'react';
import cn from 'classnames';

import { fetchUsersSearch } from 'entities/User/api/fetchUsersSearch/fetchUsersSearch';
import { UserStatus } from 'entities/User/consts/UserStatus';
import { getUserFullName } from 'entities/User/helpers/getUserFullName/getUserFullName';
import { getUserSetupEmailAddress } from 'entities/User/helpers/getUserSetupEmailAddress/getUserSetupEmailAddress';
import { getUserSetupPhoneNumber } from 'entities/User/helpers/getUserSetupPhoneNumber/getUserSetupPhoneNumber';
import { UserSearchSchema } from 'entities/User/types/UserSearchSchema';
import { UserShortSchema } from 'entities/User/types/UserShortSchema';
import { UserStatusForm } from 'entities/User/ui/UserStatusForm/UserStatusForm';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { InputSize } from 'shared/consts/InputSize';
import { Path } from 'shared/consts/Path';
import { formatPhoneNumber } from 'shared/helpers/formatPhoneNumber/formatPhoneNumber';
import { CacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { DotText } from 'shared/ui/DotText/DotText';
import { Dropdown } from 'shared/ui/Dropdown/Dropdown';
import { InputEmail } from 'shared/ui/InputEmail/InputEmail';
import { InputPhoneNumber } from 'shared/ui/InputPhoneNumber/InputPhoneNumber';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { Suggest } from 'shared/ui/Suggest/Suggest';

import { i18n } from 'entities/User/ui/UserInput/UserInput.i18n';

import MoreIcon from 'shared/ui/Icons/images/more-horizontal-24.inline.svg';

import styles from 'entities/User/ui/UserInput/UserInput.css';

export type UserInputData = OneOf<[UserInputState, UserShortSchema]>;

export type UserInputErrors = Record<keyof UserInputState | 'required', string>;

export interface UserInputProps {
    className?: string;
    initialValue?: UserShortSchema;
    disableUserSwitch?: boolean;
    error?: FormErrorSchema<keyof UserInputState | string>;

    onUserInputChange?(payload: UserInputData): void;
}

export interface UserInputState {
    first_name: string;
    last_name: string;
    phone_number: string;
    email?: string;
}

const EMPTY_USER = {
    first_name: '',
    last_name: '',
    phone_number: '',
    email: '',
};

// @todo: add storybook test
export const UserInput: React.FC<UserInputProps> = function UserInput({
    className,
    initialValue,
    disableUserSwitch,
    error,
    onUserInputChange,
}) {
    const errors: UserInputErrors = React.useMemo(() => {
        if (!error) {
            return {} as UserInputErrors;
        }

        return error.reduce(
            (obj, item) => ({
                ...obj,
                [item.code]: item.message,
            }),
            {} as UserInputErrors,
        );
    }, [error]);

    const initUser = React.useMemo<Optional<UserSearchSchema>>(() => {
        if (!initialValue?.id) {
            return undefined;
        }

        return {
            id: initialValue.id,
            status: initialValue.status,
            first_name: initialValue.first_name,
            last_name: initialValue.last_name,
            setup: {
                phone: { number: initialValue.phone_number },
                email: { address: initialValue.email || '' },
            },
        };
    }, [initialValue]);

    const [user, setUser] = React.useState<Optional<UserSearchSchema>>(initUser);
    const userState = React.useRef<UserInputState>({ ...EMPTY_USER });

    const onChange = React.useCallback(
        (userState: UserInputState, user: Optional<UserSearchSchema>) => {
            if (onUserInputChange) {
                let payload: UserInputData = userState;

                if (user) {
                    payload = {
                        id: user.id,
                        status: user.status,
                        first_name: user.first_name,
                        last_name: user.last_name,
                        phone_number: getUserSetupPhoneNumber(user.setup),
                        email: getUserSetupEmailAddress(user.setup),
                    };
                }

                onUserInputChange(payload);
            }
        },
        [onUserInputChange],
    );

    const buildChangeCallback = React.useCallback(
        (key: keyof UserInputState) => {
            return (text: string, user: Optional<UserSearchSchema>) => {
                userState.current = { ...userState.current, [key]: text };
                setUser(user);
                onChange(userState.current, user);
            };
        },
        [setUser, userState, onChange],
    );

    const buildDataProvider = React.useCallback(
        (key: keyof UserInputState, ignoreRegExp?: RegExp) => {
            return (text: string, cacheContext: CacheRequestContext) => {
                const payload: UserInputState = {
                    ...userState.current,
                    [key]: ignoreRegExp ? text.replace(ignoreRegExp, '') : text,
                };

                const search = Object.values(payload).filter(Boolean).join(',');

                return fetchUsersSearch(search, cacheContext);
            };
        },
        [userState],
    );

    const dataProviderFirstName = React.useCallback(buildDataProvider('first_name'), [buildDataProvider]);
    const dataProviderLastName = React.useCallback(buildDataProvider('last_name'), [buildDataProvider]);
    const dataProvidePhoneNumber = React.useCallback(buildDataProvider('phone_number', /^\+/g), [buildDataProvider]);
    const dataProviderEmail = React.useCallback(buildDataProvider('email'), [buildDataProvider]);

    const onChangeFirstName = React.useCallback(buildChangeCallback('first_name'), [buildChangeCallback]);
    const onChangeLastName = React.useCallback(buildChangeCallback('last_name'), [buildChangeCallback]);
    const onChangeEmail = React.useCallback(buildChangeCallback('email'), [buildChangeCallback]);
    const onChangePhoneNumber = React.useCallback(buildChangeCallback('phone_number'), [buildChangeCallback]);

    const onSwitchUserHandler = React.useCallback(() => {
        userState.current = { ...EMPTY_USER };
        setUser(undefined);
        onChange(userState.current, undefined);
    }, [userState]);

    const menuItemsProvider = React.useCallback((items: UserSearchSchema[]) => {
        return items
            .filter((item) => item.first_name.length)
            .map((item) => {
                const { id, first_name: firstName, last_name: lastName, setup } = item;

                const phoneNumber = formatPhoneNumber(getUserSetupPhoneNumber(setup));
                const email = getUserSetupEmailAddress(setup);
                const fullName = getUserFullName(firstName, lastName);

                return {
                    id,
                    text: (fullName + ' ' + phoneNumber).trim() || email,
                    data: item,
                    content: (
                        <span className={styles.menu}>
                            <span className={styles.name}>{fullName}</span>

                            <DotText
                                className={styles.details}
                                items={[phoneNumber, email]}
                            />
                        </span>
                    ),
                };
            });
    }, []);

    const moreMenuItems = React.useMemo((): MenuItemOptions[] => {
        const switchUser = {
            value: 'switch-customer',
            label: i18n('Switch into other customer'),
            onClick: onSwitchUserHandler,
        };

        return [
            !disableUserSwitch ? switchUser : null,
            user?.status !== UserStatus.DELETED
                ? {
                      value: 'open-profile',
                      label: i18n('Open user profile in new window'),
                      href: `${Path.USERS}/${user?.id}/profile`,
                      target: '_blank',
                  }
                : null,
        ].filter(Boolean) as MenuItemOptions[];
    }, [user, onSwitchUserHandler]);

    return (
        <div className={className}>
            {!user && (
                <div className={cn(styles.item, styles.user)}>
                    <Suggest<UserSearchSchema>
                        inputSize={InputSize.M}
                        placeholder={i18n('Name')}
                        hasError={Boolean(errors.required || errors.first_name)}
                        hasClear
                        hasArrow={false}
                        onSuggestChange={onChangeFirstName}
                        dataProvider={dataProviderFirstName}
                        menuItemsProvider={menuItemsProvider}
                    />

                    <Suggest<UserSearchSchema>
                        inputSize={InputSize.M}
                        placeholder={i18n('Surname')}
                        hasArrow={false}
                        hasError={Boolean(errors.required || errors.last_name)}
                        hasClear
                        onSuggestChange={onChangeLastName}
                        dataProvider={dataProviderLastName}
                        menuItemsProvider={menuItemsProvider}
                    />

                    <Suggest<UserSearchSchema>
                        inputComponent={InputPhoneNumber}
                        inputSize={InputSize.M}
                        placeholder={i18n('Phone number')}
                        hasArrow={false}
                        hasError={Boolean(errors.required || errors.phone_number)}
                        hasClear
                        onSuggestChange={onChangePhoneNumber}
                        dataProvider={dataProvidePhoneNumber}
                        menuItemsProvider={menuItemsProvider}
                    />

                    <Suggest<UserSearchSchema>
                        inputComponent={InputEmail}
                        inputSize={InputSize.M}
                        placeholder={i18n('E-mail')}
                        hasArrow={false}
                        hasError={Boolean(errors.email)}
                        hasClear
                        onSuggestChange={onChangeEmail}
                        dataProvider={dataProviderEmail}
                        menuItemsProvider={menuItemsProvider}
                    />
                </div>
            )}

            {user && (
                <UserStatusForm
                    fullName={getUserFullName(user.first_name, user.last_name)}
                    phoneNumber={getUserSetupPhoneNumber(user.setup)}
                    email={getUserSetupEmailAddress(user.setup)}
                    status={user.status}
                    controls={
                        moreMenuItems.length ? (
                            <Dropdown
                                size={ButtonSize.M}
                                color={ButtonColor.CLEAR}
                                icon={MoreIcon}
                                items={moreMenuItems}
                                hasArrow={false}
                                position="bottom-end"
                            />
                        ) : null
                    }
                />
            )}
        </div>
    );
};
