import * as React from 'react';
import cn from 'classnames/bind';

import { EMPTY_DATA } from 'constants/constants';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { UserStatus } from 'entities/User/consts/UserStatus';

import { getDateString } from 'shared/helpers/getDateString/getDateString';

import { i18n } from 'entities/User/ui/UserStatusInfo/UserStatusInfo.i18n';

import BanIcon from 'shared/ui/Icons/images/ban-16.inline.svg';
import RecentIcon from 'shared/ui/Icons/images/recent-outline-16.inline.svg';

import styles from 'entities/User/ui/UserStatusInfo/UserStatusInfo.css';

export interface UserStatusInfoProps {
    className?: string;
    status?: UserStatus;
    withoutIcon?: boolean;
    date?: Date;
}

const cx = cn.bind(styles);

const ICON: Record<UserStatus, React.ReactNode> = {
    [UserStatus.ACTIVE]: null,
    [UserStatus.BLOCKED]: <BanIcon />,
    [UserStatus.DELETED]: null,
    [UserStatus.EXTERNAL]: <RecentIcon />,
    [UserStatus.ONBOARDING]: <RecentIcon />,
    [UserStatus.REJECTED]: <BanIcon />,
    [UserStatus.SCREENING]: <RecentIcon />,
};

export const UserStatusInfo: React.FC<UserStatusInfoProps> = function UserStatusInfo({
    className,
    status = UserStatus.EXTERNAL,
    withoutIcon,
    date,
}) {
    const locale = getI18nLocale();

    const label: Record<UserStatus, string> = React.useMemo(
        () => ({
            [UserStatus.ACTIVE]: i18n('Active'),
            [UserStatus.BLOCKED]: i18n('Blocked'),
            [UserStatus.DELETED]: i18n('Deleted'),
            [UserStatus.EXTERNAL]: i18n('Waiting for activation'),
            [UserStatus.ONBOARDING]: i18n('Onboarding'),
            [UserStatus.REJECTED]: i18n('Rejected'),
            [UserStatus.SCREENING]: i18n('Processing documents'),
        }),

        [],
    );

    return (
        <div className={cx(styles.status, [status, className])}>
            {/*@TODO использовать TextWithDot*/}
            {!withoutIcon && label[status] && <span className={styles.icon}>{ICON[status]}</span>}

            {label[status] || EMPTY_DATA}

            {date && (
                <>
                    {' '}
                    {i18n('since {date}', {
                        date: getDateString(date, locale, {
                            day: 'numeric',
                            month: 'long',
                            year: 'numeric',
                        }),
                    })}
                </>
            )}
        </div>
    );
};
