import * as React from 'react';

import { useCarOfferConfig } from 'features/CarBooking/api/useCarOfferConfig/useCarOfferConfig';
import { CarBookingForm } from 'features/CarBooking/ui/CarBookingForm/CarBookingForm';

import { postCarRentalForm } from 'entities/Car/api/postCarRentalForm/postCarRentalForm';
import { CarOfferFormSchema } from 'entities/Car/types/CarOfferFormSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { ModalContainer } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/CarBooking/ui/CarAddBookingModal/CarAddBookingModal.i18n';

export interface CarAddBookingModalProps {
    initial?: Partial<CarOfferFormSchema>;

    onChange?(): void;

    onClose?(): void;
}

export const CarAddBookingModal: React.FC<CarAddBookingModalProps> = function CarAddBookingModal({
    initial,
    onChange,
    onClose,
}) {
    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<CarOfferFormSchema>>();

    const { data, isLoading } = useCarOfferConfig();

    const onSubmit = React.useCallback(async () => {
        if (validate()) {
            const data = getValues();

            try {
                await postCarRentalForm(data as CarOfferFormSchema);
            } catch (error) {
                return setError('_serverError', getFetchErrorMessage(error));
            }

            if (onChange) {
                onChange();
            }

            if (onClose) {
                onClose();
            }
        }
    }, [onClose, onChange, getValues, validate]);

    if (!data || isLoading) {
        return null;
    }

    return (
        <ModalContainer
            title={i18n('Add booking')}
            hasClose
            closeOnOverlay={false}
            controls={
                <Button
                    size={ButtonSize.M}
                    color={ButtonColor.PRIMARY}
                    label={i18n('Book')}
                    onClick={onSubmit}
                />
            }
            onClose={onClose}
        >
            <CarBookingForm
                initial={initial}
                controller={controller}
                offerConfig={data}
            />
        </ModalContainer>
    );
};
