import * as React from 'react';
import cn from 'classnames/bind';

import { DEFAULT_BOOKING_TIME } from 'features/CarBooking/consts/constants';
import { validateUserInput } from 'features/CarBooking/helpers/validateUserInput/validateUserInput';
import { CarBookingOfferOptions } from 'features/CarBooking/ui/CarBookingOfferOptions/CarBookingOfferOptions';

import { CarInput, getCarOfferCurrencyName } from 'entities/Car';
import { CarOfferConfigSchema } from 'entities/Car/types/CarOfferConfigSchema';
import { CarOfferFormSchema } from 'entities/Car/types/CarOfferFormSchema';
import { CarOfferSessionSchema } from 'entities/Car/types/CarOfferSessionSchema';
import { CarOfferStatusSchema } from 'entities/Car/types/CarOfferStatusSchema';
import { GeoInput } from 'entities/Geo';
import { FetchGeoSuggestOptions } from 'entities/Geo/api/fetchGeoSuggest/fetchGeoSuggest';
import { UserInput } from 'entities/User';
import { UserShortSchema } from 'entities/User/types/UserShortSchema';

import { ButtonSize } from 'shared/consts/ButtonSize';
import { InputSize } from 'shared/consts/InputSize';
import { TextareaSize } from 'shared/consts/TextareaSize';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { validatePositiveNumber } from 'shared/helpers/validatePositiveNumber/validatePositiveNumber';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormDateTimeRange } from 'shared/ui/FormDateTimeRange/FormDateTimeRange';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { InputDescription } from 'shared/ui/InputDescription/InputDescription';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { RadioButton, RadioButtonItemOptions } from 'shared/ui/RadioButton/RadioButton';
import { Select } from 'shared/ui/Select/Select';
import { Textarea } from 'shared/ui/Textarea/Textarea';

import { i18n } from 'features/CarBooking/ui/CarBookingForm/CarBookingForm.i18n';

import styles from 'features/CarBooking/ui/CarBookingForm/CarBookingForm.css';

export interface CarBookingFormProps {
    className?: string;

    initial?: Partial<CarOfferFormSchema>;
    session?: CarOfferSessionSchema;

    offerConfig: CarOfferConfigSchema;

    controller?: UseFormControllerSub<CarOfferFormSchema>;

    disableUserSwitch?: boolean;
    disableInProgress?: boolean;
    disableCompleted?: boolean;

    onFormChange?(state: CarOfferFormSchema): void;
}

const cx = cn.bind(styles);

// @todo: use getCarOfferStatus instead
const CAR_STATUS: RadioButtonItemOptions<CarOfferStatusSchema>[] = [
    { label: i18n('Draft'), id: 'draft' },
    { label: i18n('Confirmed'), id: 'confirmed' },
    { label: i18n('Paid'), id: 'paid' },
];

// TODO: move to offerConfig
const GEO_INPUT_OPTIONS: Partial<FetchGeoSuggestOptions> = {
    countries: 'cz',
    lang: 'en_US',
};

export const CarBookingForm: React.FC<CarBookingFormProps> = function CarBookingForm({
    className,
    initial,
    session,
    offerConfig,
    controller,
    disableUserSwitch,
    disableInProgress,
    disableCompleted,
    onFormChange,
}) {
    const hasOverrunOptions =
        typeof offerConfig.overrun_cost_per_km === 'number' || typeof offerConfig.limit_km_per_day === 'number';

    const validationRules: UseFormValidation<OptionalRecord<CarOfferFormSchema>> = React.useMemo(() => {
        return {
            user: validateUserInput(),
            car: validateRequired(i18n('Select car')),

            since: validateRequired(i18n('Enter date and time')), // todo validate date
            since_time: validateRequired(i18n('Enter date and time')), // todo validate time
            until: validateRequired(i18n('Enter date and time')), // todo validate date
            until_time: validateRequired(i18n('Enter date and time')), // todo validate time

            delivery_location: validateRequired(i18n('Select pick-up point')),
            return_location: validateRequired(i18n('Select return point')),

            total_payment: validatePositiveNumber(i18n('Enter total amount')),
            deposit: validatePositiveNumber(i18n('Enter deposit')),

            ...(hasOverrunOptions
                ? {
                      limit_km_per_day: validatePositiveNumber(i18n('Enter kilometers included')),
                      overrun_cost_per_km: validatePositiveNumber(i18n('Enter extra mileage charge')),
                  }
                : {}),
        };
    }, [hasOverrunOptions]);

    const carInsuranceList: RadioButtonItemOptions<CarOfferFormSchema['insurance_type']>[] = React.useMemo(() => {
        if (!offerConfig.insurance_types || !offerConfig.insurance_types.length) {
            return [];
        }

        return offerConfig.insurance_types.map(({ title, id }) => ({
            id,
            label: title,
        }));
    }, [offerConfig]);

    const currencies: MenuItemOptions<CarOfferFormSchema['currency']>[] = React.useMemo(() => {
        if (!offerConfig.currencies || !offerConfig.currencies.length) {
            return [];
        }

        return offerConfig.currencies.map(({ title, id }) => ({
            value: id,
            label: title,
        }));
    }, [offerConfig]);

    const init = React.useMemo(() => {
        const initForm: OptionalRecord<CarOfferFormSchema> = {
            user: undefined,
            car: undefined,

            offer_options: undefined,

            insurance_type: carInsuranceList.length ? carInsuranceList[0].id : undefined,

            since: undefined,
            since_time: DEFAULT_BOOKING_TIME,
            until: undefined,
            until_time: DEFAULT_BOOKING_TIME,

            delivery_location: undefined,
            return_location: undefined,

            comment: '',
            total_payment: '',
            deposit: '',
            currency: currencies.length ? currencies[0].value : '',

            limit_km_per_day: offerConfig.limit_km_per_day || '',
            overrun_cost_per_km: offerConfig.overrun_cost_per_km || '',

            status: 'draft',
        };

        return {
            ...initForm,
            ...initial,
        };
    }, [offerConfig, initial, carInsuranceList, currencies]);

    const { update, watch, getValue, errors } = useForm<OptionalRecord<CarOfferFormSchema>>({
        init,
        validation: validationRules,
        controller,
        onFormChange,
    });

    const today = React.useMemo(() => getDateStartDay(new Date()), []);

    return (
        <FormContainer className={className}>
            <FormLabel title={i18n('Customer')}>
                <FormErrorContainer error={<ErrorLabel errors={errors.user} />}>
                    <UserInput
                        initialValue={getValue('user') as UserShortSchema}
                        disableUserSwitch={disableUserSwitch}
                        error={errors.user}
                        onUserInputChange={update('user')}
                    />
                </FormErrorContainer>
            </FormLabel>

            <FormLabel title={i18n('Car')}>
                <FormErrorContainer error={<ErrorLabel errors={errors.car} />}>
                    <CarInput
                        inputSize={InputSize.M}
                        placeholder={i18n('Select')}
                        disabled={disableInProgress || disableCompleted}
                        hasError={Boolean(errors.car)}
                        initialValue={getValue('car')}
                        onCarInputChange={update('car')}
                    />
                </FormErrorContainer>
            </FormLabel>

            {Boolean(offerConfig.offer_options && offerConfig.offer_options.length) && (
                <FormLabel title={i18n('Options')}>
                    <CarBookingOfferOptions
                        className={cx(styles.item, styles.options, { half: true })}
                        offerOptions={offerConfig.offer_options!}
                        disableCompleted={disableCompleted}
                        values={watch('offer_options')}
                        onChange={update('offer_options')}
                    />
                </FormLabel>
            )}

            {Boolean(carInsuranceList.length) && (
                <FormLabel title={i18n('Insurance')}>
                    <RadioButton<CarOfferFormSchema['insurance_type']>
                        className={styles.radioItem}
                        items={carInsuranceList}
                        value={watch('insurance_type') || carInsuranceList[0].id}
                        disabled={disableCompleted}
                        onChange={update('insurance_type')}
                    />
                </FormLabel>
            )}

            <FormDateTimeRange<CarOfferFormSchema>
                title={i18n('Dates')}
                actualSince={session?.actual_since}
                actualUntil={session?.actual_until}
                minBookingDate={today}
                disableSince={disableInProgress || disableCompleted}
                disableUntil={disableCompleted}
                update={update}
                watch={watch}
                errors={errors}
            />

            <FormLabel title={i18n('Pick-up and return points')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.delivery_location} />}>
                        <GeoInput
                            inputSize={InputSize.M}
                            placeholder={i18n('Pick-up point')}
                            disabled={disableInProgress || disableCompleted}
                            hasError={Boolean(errors.delivery_location)}
                            hasClear
                            initialValue={getValue('delivery_location')}
                            geoSuggestOptions={GEO_INPUT_OPTIONS}
                            defaultItems={offerConfig.delivery_locations}
                            onGeoInputChange={update('delivery_location')}
                        />
                    </FormErrorContainer>

                    <FormErrorContainer error={<ErrorLabel errors={errors.return_location} />}>
                        <GeoInput
                            inputSize={InputSize.M}
                            placeholder={i18n('Return point')}
                            disabled={disableCompleted}
                            hasError={Boolean(errors.return_location)}
                            hasClear
                            initialValue={getValue('return_location')}
                            geoSuggestOptions={GEO_INPUT_OPTIONS}
                            defaultItems={offerConfig.return_locations}
                            onGeoInputChange={update('return_location')}
                        />
                    </FormErrorContainer>
                </div>
            </FormLabel>

            <FormLabel title={i18n('Comment')}>
                <Textarea
                    textareaSize={TextareaSize.M}
                    expandable
                    placeholder={i18n('Customer will not see this')}
                    value={watch('comment')}
                    disabled={disableCompleted}
                    hasClear
                    onTextareaChange={update('comment')}
                />
            </FormLabel>

            <div className={cn(styles.item, styles.payment)}>
                <FormLabel title={i18n('Total amount')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.total_payment} />}>
                        <Input
                            inputSize={InputSize.M}
                            inputMode="numeric"
                            type="number"
                            value={watch('total_payment')}
                            disabled={disableCompleted}
                            hasError={Boolean(errors.total_payment)}
                            hasClear
                            onInputChange={update('total_payment')}
                        />
                    </FormErrorContainer>
                </FormLabel>

                <FormLabel title={i18n('Deposit')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.deposit} />}>
                        <Input
                            inputSize={InputSize.M}
                            inputMode="numeric"
                            type="number"
                            value={watch('deposit')}
                            disabled={disableCompleted}
                            hasError={Boolean(errors.deposit)}
                            hasClear
                            onInputChange={update('deposit')}
                        />
                    </FormErrorContainer>
                </FormLabel>

                <FormLabel title={i18n('Currency')}>
                    <Select
                        size={ButtonSize.M}
                        items={currencies}
                        checkedMenu={watch('currency')}
                        disabled={disableCompleted}
                        onSelectChange={update('currency') as any}
                    />
                </FormLabel>
            </div>

            {hasOverrunOptions && (
                <div className={cx(styles.item, { half: true })}>
                    <FormLabel title={i18n('Kilometers included')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.limit_km_per_day} />}>
                            <InputDescription
                                inputSize={InputSize.M}
                                inputMode="numeric"
                                type="number"
                                value={watch('limit_km_per_day')}
                                description={i18n('km/day')}
                                disabled={disableCompleted}
                                hasError={Boolean(errors.limit_km_per_day)}
                                hasClear
                                onInputChange={update('limit_km_per_day')}
                            />
                        </FormErrorContainer>
                    </FormLabel>

                    <FormLabel title={i18n('Extra mileage charge')}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.overrun_cost_per_km} />}>
                            <InputDescription
                                inputSize={InputSize.M}
                                inputMode="numeric"
                                type="number"
                                value={watch('overrun_cost_per_km')}
                                description={`${getCarOfferCurrencyName(
                                    watch('currency')!,
                                    offerConfig.currencies,
                                )} ${i18n('per 1 km')}`}
                                disabled={disableCompleted}
                                hasError={Boolean(errors.overrun_cost_per_km)}
                                hasClear
                                onInputChange={update('overrun_cost_per_km')}
                            />
                        </FormErrorContainer>
                    </FormLabel>
                </div>
            )}

            <FormLabel title={i18n('Status')}>
                <RadioButton<CarOfferStatusSchema>
                    className={styles.radioItem}
                    items={CAR_STATUS}
                    value={watch('status')!}
                    disabled={disableInProgress || disableCompleted}
                    onChange={update('status')}
                />
            </FormLabel>

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
