import * as React from 'react';

import { deleteCarOffer } from 'features/CarBooking/api/deleteCarOffer/deleteCarOffer';
import { CarBookingForm } from 'features/CarBooking/ui/CarBookingForm/CarBookingForm';
import { CarCancelBookingDialog } from 'features/CarBooking/ui/CarCancelBookingDialog/CarCancelBookingDialog';

import { postCarRentalForm } from 'entities/Car/api/postCarRentalForm/postCarRentalForm';
import { useCarOffer } from 'entities/Car/api/useCarOffer/useCarOffer';
import { CarOfferSessionStatus } from 'entities/Car/consts/CarOfferSessionStatus';
import { getCarOfferSessionStatus } from 'entities/Car/helpers/getCarOfferSessionStatus/getCarOfferSessionStatus';
import { CarOfferFormSchema } from 'entities/Car/types/CarOfferFormSchema';
import { UserShortSchema } from 'entities/User/types/UserShortSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { ModalContainer } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/CarBooking/ui/CarEditBookingModal/CarEditBookingModal.i18n';

export interface CarEditBookingModalProps {
    tagId: string;
    offerId: string;
    user?: UserShortSchema;

    onChange?(): void;

    onClose?(): void;
}

export const CarEditBookingModal: React.FC<CarEditBookingModalProps> = function CarEditBookingModal({
    tagId,
    offerId,
    user,
    onChange,
    onClose,
}) {
    const { data, isLoading } = useCarOffer(offerId);

    const initial = { ...data?.offer, user };

    const [isConfirmVisible, setConfirmVisible] = React.useState<boolean>(false);

    const { isEqualData, onFormChangeHandler } = useEqualDataForm<CarOfferFormSchema>(initial);
    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<CarOfferFormSchema>>();

    const showCancelConfirm = React.useCallback(() => {
        setConfirmVisible(true);
    }, [setConfirmVisible]);

    const hideCancelConfirm = React.useCallback(() => {
        setConfirmVisible(false);
    }, [setConfirmVisible]);

    const cancelBooking = React.useCallback(async () => {
        try {
            await deleteCarOffer(tagId);

            // @todo: add notification
        } catch (error) {
            hideCancelConfirm();

            // TODO: show notification here instead error message
            return setError('_serverError', getFetchErrorMessage(error));
        }

        hideCancelConfirm();

        if (onChange) {
            onChange();
        }

        if (onClose) {
            onClose();
        }
    }, [tagId, onChange, onClose, hideCancelConfirm]);

    const submitBooking = React.useCallback(async () => {
        if (validate()) {
            const formData = getValues();

            try {
                await postCarRentalForm(formData as CarOfferFormSchema, { tagId, offerId, prev: data?.raw! });

                // @todo: add notification
            } catch (error) {
                return setError('_serverError', getFetchErrorMessage(error));
            }

            if (onChange) {
                onChange();
            }

            if (onClose) {
                onClose();
            }
        }
    }, [tagId, offerId, onChange, onClose, getValues, validate, data]);

    if (!data || isLoading) {
        return null;
    }

    const sessionStatus = getCarOfferSessionStatus(data.session);
    const disableCompleted =
        sessionStatus === CarOfferSessionStatus.COMPLETED || sessionStatus === CarOfferSessionStatus.OUTDATED;

    return (
        <>
            <ModalContainer
                title={i18n('Edit booking')}
                hasClose
                onClose={onClose}
                closeOnOverlay={false}
                controls={
                    <>
                        {disableCompleted ? (
                            <Button
                                size={ButtonSize.M}
                                color={ButtonColor.SECONDARY}
                                label={i18n('Close')}
                                onClick={onClose}
                            />
                        ) : (
                            <Button
                                size={ButtonSize.M}
                                color={ButtonColor.PRIMARY}
                                label={i18n('Save changes')}
                                disabled={isEqualData}
                                onClick={submitBooking}
                            />
                        )}

                        {sessionStatus !== CarOfferSessionStatus.IN_PROGRESS &&
                            sessionStatus !== CarOfferSessionStatus.COMPLETED && (
                                <Button
                                    size={ButtonSize.M}
                                    color={ButtonColor.ALARM}
                                    label={i18n('Cancel booking')}
                                    onClick={showCancelConfirm}
                                />
                            )}
                    </>
                }
            >
                <CarBookingForm
                    initial={initial}
                    session={data.session}
                    offerConfig={data.offerConfig}
                    controller={controller}
                    disableUserSwitch
                    disableInProgress={sessionStatus === CarOfferSessionStatus.IN_PROGRESS}
                    disableCompleted={disableCompleted}
                    onFormChange={onFormChangeHandler}
                />
            </ModalContainer>

            {isConfirmVisible && (
                <CarCancelBookingDialog
                    onDialogSubmit={cancelBooking}
                    onClose={hideCancelConfirm}
                />
            )}
        </>
    );
};
