import * as React from 'react';
import { useHistory, useParams, useRouteMatch } from 'react-router-dom';
import cn from 'classnames';

import { isCarDamageFlag } from 'utils/isCarDamageFlag';
import { isShowSessionsFlag } from 'utils/isShowSessionsFlag';
import { isShowTelematicsFlag } from 'utils/isShowTelematicsFlag';

import { UseCarResource } from 'entities/Car/api/useCar/useCar';
import { CarCardTab } from 'entities/Car/consts/CarCardTab';

import { Path } from 'shared/consts/Path';
import { TabSize } from 'shared/consts/TabSize';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { TabItemOptions } from 'shared/ui/TabItem/TabItem';
import { Tabs } from 'shared/ui/Tabs/Tabs';

import { i18n } from 'features/CarCardHeaderTabs/ui/CarCardHeaderTabs/CarCardHeaderTabs.i18n';

import styles from 'features/CarCardHeaderTabs/ui/CarCardHeaderTabs/CarCardHeaderTabs.css';

// @TODO убрать когда разьедимся на 4 страницы
const CAR_HEADER_TAB_ROUTES: Record<CarCardTab, Path> = {
    [CarCardTab.OVERVIEW]: Path.CAR_OVERVIEW,
    [CarCardTab.SIGNALS]: Path.CAR_SIGNALS,
    [CarCardTab.SESSIONS]: Path.CAR_RIDES,
    [CarCardTab.TELEMATICS]: Path.CAR_TELEMATICS,
    [CarCardTab.DAMAGES]: Path.CAR_DAMAGES,
};

export interface CarCardHeaderTabsProps {
    //currentItem: CarCardTab; @TODO
    className?: string;

    resource?: UseCarResource;
}

export const CarCardHeaderTabs: React.FC<CarCardHeaderTabsProps> = function CarCardHeaderTabs({ className, resource }) {
    // @todo: move to page
    const history = useHistory();
    const params = useParams();

    const currentItem =
        Object.keys(CAR_HEADER_TAB_ROUTES).find((key) => {
            return useRouteMatch(CAR_HEADER_TAB_ROUTES[key]);
        }) || CarCardTab.OVERVIEW;

    const car = useResourceReader(resource);

    const hasCarTelematics = Boolean(car?.telematics);
    const showTelematicsTab = hasCarTelematics && isShowTelematicsFlag();

    const onItemClickHandler = React.useCallback(
        (value: CarCardTab) => {
            history.push(generateRouterPath(CAR_HEADER_TAB_ROUTES[value], params));
        },
        [history, params],
    );

    const items = React.useMemo(() => {
        let list: TabItemOptions[] = [
            {
                label: i18n('Overview'),
                value: CarCardTab.OVERVIEW,
            },
            {
                label: i18n('Signals'),
                value: CarCardTab.SIGNALS,
            },
        ];

        if (isShowSessionsFlag()) {
            list.push({
                label: i18n('Rides'),
                value: CarCardTab.SESSIONS,
            });
        }

        if (showTelematicsTab) {
            list.push({
                label: i18n('Telematics'),
                value: CarCardTab.TELEMATICS,
            });
        }

        if (isCarDamageFlag()) {
            list.push({
                label: i18n('Damages'),
                value: CarCardTab.DAMAGES,
            });
        }

        return list;
    }, [showTelematicsTab]);

    return (
        <Tabs
            className={cn(styles.carCardHeaderTabs, className)}
            size={TabSize.M}
            items={items}
            currentItem={currentItem}
            onItemClick={onItemClickHandler}
        />
    );
};
