import * as React from 'react';
import { useEffect, useState } from 'react';

import { postCarCreate, PostCarCreatePayloadItem } from 'features/CarCreate/api/postCarCreate/postCarCreate';
import { XLSX_TYPE } from 'features/CarCreate/consts/constants';
import { generateCarsPayload } from 'features/CarCreate/helpers/generateCarsPayload';
import { processXLSX } from 'features/CarCreate/helpers/processXLSX';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Button } from 'shared/ui/Button/Button';
import { ButtonFile } from 'shared/ui/ButtonFile/ButtonFile';
import { Link } from 'shared/ui/Link/Link';
import { ModalContainer } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/CarCreate/ui/CarBulkCreateModal/CarBulkCreateModal.i18n';

import AttachIcon from 'shared/ui/Icons/images/attach-16.inline.svg';

import style from 'features/CarCreate/ui/CarBulkCreateModal/CarBulkCreateModal.css';

export interface CarBulkCreateModalProps {
    onClose(closeBoth?: boolean): void;

    onChange?(): void;
}

export const CarBulkCreateModal: React.FC<CarBulkCreateModalProps> = function CarBulkCreateModal({
    onClose,
    onChange,
}) {
    const [cars, setCars] = useState<PostCarCreatePayloadItem[]>([]);
    const [file, setFile] = useState<File | null>(null);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        file &&
            processXLSX(file)
                .then((carArrays) => {
                    const cars = generateCarsPayload(carArrays);
                    setCars(cars);
                })
                .catch(() => {
                    setError(i18n('Failed to parse data'));
                });
    }, [file]);

    const onBrowseCars = React.useCallback((values: File[]) => {
        const file: File = values?.[0];

        if (file?.type !== XLSX_TYPE) {
            setError(i18n('Incorrect file type'));
        } else {
            setError(null);
            setFile(file);
        }
    }, []);

    const createCars = React.useCallback(async () => {
        try {
            await postCarCreate(cars);

            onChange?.();
            onClose(true);
        } catch (error) {
            setError(i18n('Something went wrong'));
        }
    }, [onClose, onChange, cars]);

    const onModalClose = React.useCallback(() => {
        onClose(false);
    }, [onClose]);

    return (
        <ModalContainer
            title={i18n('Add cars via XLSX-file')}
            hasClose
            onClose={onModalClose}
            controls={[
                <Button
                    size={ButtonSize.M}
                    color={ButtonColor.PRIMARY}
                    key="AddCarsButton"
                    label={i18n('Add cars')}
                    disabled={!cars.length}
                    onClick={createCars}
                />,
                <Button
                    size={ButtonSize.M}
                    color={ButtonColor.SECONDARY}
                    key="BackToManualButton"
                    label={i18n('Back to manual adding')}
                    onClick={onClose?.bind(null, false)}
                />,
            ]}
        >
            <div>
                <div className={style.subtitle}>{i18n('Car details')}</div>

                <ButtonFile
                    className={style.browseButton}
                    accept=".xlsx"
                    color={ButtonColor.SECONDARY}
                    size={ButtonSize.M}
                    icon={AttachIcon}
                    label={file?.name || i18n('Browse XLSX-file')}
                    onInputChange={onBrowseCars}
                />

                {error ? <div className={style.error_label}>{error}</div> : null}

                <div className={style.description}>
                    {i18n(
                        'To upload several cars at once {link} and fill it out with your data. Then upload your XLSX-file using this form',
                        {
                            link: (
                                <Link
                                    href="https://drivematics.s3.yandex.net/tmpl/CarUploadTemplate.xlsx"
                                    target="_blank"
                                    download="CarUploadTemplate.xlsx"
                                >
                                    {i18n('download the template')}
                                </Link>
                            ),
                        },
                        { __raw__: true },
                    )}
                </div>
            </div>
        </ModalContainer>
    );
};
