import * as React from 'react';

import { CarCreateFormSchema } from 'features/CarCreate/types/CarCreateFormSchema';

import { CarModelInput } from 'entities/Car/ui/CarModelInput/CarModelInput';

import { InputSize } from 'shared/consts/InputSize';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';

import { i18n } from 'features/CarCreate/ui/CarCreateForm/CarCreateForm.i18n';

import styles from 'features/CarCreate/ui/CarCreateForm/CarCreateForm.css';

export interface CarCreateFormProps {
    className?: string;

    controller?: UseFormControllerSub<OptionalRecord<CarCreateFormSchema>>;
}

const VALIDATION_RULES: UseFormValidation<OptionalRecord<CarCreateFormSchema>> = {
    vin: validateRequired(),
    number: validateRequired(),
    model: validateRequired(),
};

const INIT_FORM: OptionalRecord<CarCreateFormSchema> = {
    vin: '',
    number: '',
    model: undefined,
};

const MENU_MAX_HEIGHT = 190;

export const CarCreateForm: React.FC<CarCreateFormProps> = function CarCreateForm({ className, controller }) {
    const { update, errors } = useForm<OptionalRecord<CarCreateFormSchema>>({
        init: { ...INIT_FORM },
        validation: VALIDATION_RULES,
        controller,
    });

    return (
        <FormContainer className={className}>
            <FormLabel title={i18n('Car ID')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.vin} />}>
                        <Input
                            inputSize={InputSize.M}
                            placeholder={i18n('VIN')}
                            hasClear
                            onInputChange={update('vin')}
                            hasError={Boolean(errors.vin)}
                        />
                    </FormErrorContainer>

                    <FormErrorContainer error={<ErrorLabel errors={errors.number} />}>
                        <Input
                            inputSize={InputSize.M}
                            placeholder={i18n('License plate number')}
                            hasClear
                            onInputChange={update('number')}
                            hasError={Boolean(errors.number)}
                        />
                    </FormErrorContainer>
                </div>
            </FormLabel>

            <FormLabel title={i18n('Car details')}>
                <FormErrorContainer error={<ErrorLabel errors={errors.model} />}>
                    <CarModelInput
                        placeholder={i18n('Model')}
                        onCarModelInputChange={update('model')}
                        hasError={Boolean(errors.model)}
                        inputSize={InputSize.M}
                        maxHeight={MENU_MAX_HEIGHT}
                    />
                </FormErrorContainer>
            </FormLabel>

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
