import * as React from 'react';
import { useState } from 'react';
import cn from 'classnames';

import { postCarCreate } from 'features/CarCreate/api/postCarCreate/postCarCreate';
import { CarCreateFormSchema } from 'features/CarCreate/types/CarCreateFormSchema';
import { CarBulkCreateModal } from 'features/CarCreate/ui/CarBulkCreateModal/CarBulkCreateModal';
import { CarCreateForm } from 'features/CarCreate/ui/CarCreateForm/CarCreateForm';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { Link } from 'shared/ui/Link/Link';
import { ModalContainer } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/CarCreate/ui/CarCreateModal/CarCreateModal.i18n';

import style from 'features/CarCreate/ui/CarCreateModal/CarCreateModal.css';

export interface CarCreateModalProps {
    className?: string;
    onClose?(): void;
    onChange?(): void;
}

export const CarCreateModal: React.FC<CarCreateModalProps> = function CarCreateModal({ className, onClose, onChange }) {
    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<CarCreateFormSchema>>();

    const [isBulkUploadOpen, setBulkUploadOpen] = useState<boolean>(false);

    const onBulkUploadOpen = React.useCallback(() => {
        setBulkUploadOpen(true);
    }, [setBulkUploadOpen]);

    const onBulkUploadClose = React.useCallback(
        (closeBoth?: boolean) => {
            setBulkUploadOpen(false);

            if (closeBoth && onClose) {
                onClose();
            }
        },
        [setBulkUploadOpen, onClose],
    );

    const onSubmit = React.useCallback(async () => {
        if (validate()) {
            const data = getValues();

            try {
                await postCarCreate(data as CarCreateFormSchema);
            } catch (error) {
                return setError('_serverError', getFetchErrorMessage(error));
            }

            if (onChange) {
                onChange();
            }

            if (onClose) {
                onClose();
            }
        }
    }, [onClose, onChange, getValues, validate, setError]);

    return (
        <ModalContainer
            className={cn(className, style.carCreateContainer)}
            title={i18n('Add car')}
            hasClose
            onClose={onClose}
            controls={
                <Button
                    size={ButtonSize.M}
                    color={ButtonColor.PRIMARY}
                    label={i18n('Add car')}
                    className={style.addCarButton}
                    onClick={onSubmit}
                />
            }
        >
            <div className={style.multipleAdding}>
                {i18n(
                    '{link} to add several cars at once via XLSX-file',
                    {
                        link: <Link onClick={onBulkUploadOpen}>{i18n('Click here')}</Link>,
                    },
                    { __raw__: true },
                )}
            </div>

            {isBulkUploadOpen && (
                <CarBulkCreateModal
                    onClose={onBulkUploadClose}
                    onChange={onChange}
                />
            )}

            <CarCreateForm controller={controller} />
        </ModalContainer>
    );
};
