import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';
import cn from 'classnames/bind';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { CarDamagesPreviewSchema } from 'features/CarDamages/model/CarDamagesPreviewSchema';
import { CarDamagesItem } from 'features/CarDamages/ui/CarDamagesItem/CarDamagesItem';
import { CarDamagesPreview } from 'features/CarDamages/ui/CarDamagesPreview/CarDamagesPreview';

import { UseCarDamagesRes } from 'entities/Car/api/useCarDamage/useCarDamages';

import { UseFetchResource } from 'shared/hooks/useFetch/useFetch';
import { Widget } from 'shared/ui/Widget/Widget';

import { ModalObjectTypes } from 'components/types';

import { i18n } from 'features/CarDamages/ui/CarDamages/CarDamages.i18n';

import styles from 'features/CarDamages/ui/CarDamages/CarDamages.css';

export interface CarDamagesProps {
    className?: string;
    resource: UseFetchResource<UseCarDamagesRes>;
}

const cx = cn.bind(styles);

export const CarDamages: React.FC<CarDamagesProps> = function CarDamages({ className, resource }) {
    const location = useLocation();
    const history = useHistory();

    const [currentItem, setCurrentItem] = React.useState<Nullable<CarDamagesPreviewSchema>>(null);

    const damages = resource.read();

    const isEmpty = !damages || !damages.length;

    const onItemClickHandler = React.useCallback((data: CarDamagesPreviewSchema) => {
        setCurrentItem(data);
    }, []);

    const onCloseHandler = React.useCallback(() => {
        setCurrentItem(null);
    }, []);

    const onRideClickHandler = React.useCallback(
        (id: string) => {
            if (currentItem) {
                onCloseHandler();
            }

            const searchParams = new URLSearchParams(location.search);

            searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SESSION);
            searchParams.set(MODAL_OBJECT_ID_CGI, id ?? '');
            history.push(`${location.pathname}?${searchParams}`);
        },
        [location, history, currentItem, onCloseHandler],
    );

    return (
        <>
            <Widget
                className={cx(styles.content, { empty: isEmpty }, [className])}
                contentContainer
            >
                {isEmpty ? (
                    <h3 className={styles.title}>{i18n('There are no known damages for this car')}</h3>
                ) : (
                    <>
                        <h3 className={styles.title}>{i18n('{count} known damage', { count: damages.length })}</h3>

                        <div className={styles.list}>
                            {damages.map(({ url, title, created_at, user_session_id, image_id }) => (
                                <CarDamagesItem
                                    className={styles.item}
                                    src={url}
                                    title={title}
                                    createdAt={created_at}
                                    sessionId={user_session_id}
                                    onImageClick={onItemClickHandler}
                                    onRideClick={onRideClickHandler}
                                    key={image_id}
                                />
                            ))}
                        </div>
                    </>
                )}
            </Widget>

            {currentItem && (
                <CarDamagesPreview
                    {...currentItem}
                    onClose={onCloseHandler}
                    onRideClick={onRideClickHandler}
                />
            )}
        </>
    );
};
