import * as React from 'react';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { FORMAT_DATE } from 'features/CarDamages/consts/constants';
import { CarDamagesPreviewSchema } from 'features/CarDamages/model/CarDamagesPreviewSchema';

import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { Link } from 'shared/ui/Link/Link';
import { Picture } from 'shared/ui/Picture/Picture';

import { i18n } from 'features/CarDamages/ui/CarDamagesItem/CarDamagesItem.i18n';

import styles from 'features/CarDamages/ui/CarDamagesItem/CarDamagesItem.css';

export interface CarDamagesItemProps extends CarDamagesPreviewSchema {
    className?: string;

    onImageClick?(data: CarDamagesPreviewSchema): void;

    onRideClick?(id: string): void;
}

export const CarDamagesItem: React.FC<CarDamagesItemProps> = function CarDamagesItem({
    className,
    src,
    title,
    createdAt,
    sessionId,
    onImageClick,
    onRideClick,
}) {
    const locale = getI18nLocale();

    const onImageClickHandler = React.useCallback(() => {
        if (onImageClick) {
            onImageClick({ src, title, createdAt, sessionId });
        }
    }, [src, title, createdAt, sessionId, onImageClick]);

    const onRideClickHandler = React.useCallback(() => {
        if (onRideClick && sessionId) {
            onRideClick(sessionId);
        }
    }, [sessionId, onRideClick]);

    return (
        <div className={className}>
            <div className={styles.linkWrap}>
                <Link
                    className={styles.link}
                    title={title}
                    onClick={onImageClickHandler}
                />

                <Picture
                    className={styles.picture}
                    src={src}
                    title={title}
                />

                <div className={styles.details}>
                    <h5 className={styles.title}>{title}</h5>

                    {createdAt && (
                        <p className={styles.description}>
                            {i18n('Registered on {date}', {
                                date: getDateString(createdAt, locale, FORMAT_DATE),
                            })}
                        </p>
                    )}
                </div>
            </div>

            {sessionId && (
                <Link
                    className={styles.rideLink}
                    onClick={onRideClickHandler}
                >
                    {i18n('View ride')}
                </Link>
            )}
        </div>
    );
};
