import * as React from 'react';
import cn from 'classnames';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { FORMAT_DATE } from 'features/CarDamages/consts/constants';
import { CarDamagesPreviewSchema } from 'features/CarDamages/model/CarDamagesPreviewSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { usePortal } from 'shared/hooks/usePortal/usePortal';
import { usePreventScroll } from 'shared/hooks/usePreventScroll/usePreventScroll';
import { Button } from 'shared/ui/Button/Button';

import { i18n } from 'features/CarDamages/ui/CarDamagesPreview/CarDamagesPreview.i18n';

import CloseIcon from 'shared/ui/Icons/images/close-24.inline.svg';

import styles from 'features/CarDamages/ui/CarDamagesPreview/CarDamagesPreview.css';

export interface CarDamagesPreviewProps extends CarDamagesPreviewSchema {
    className?: string;

    onClose?(): void;

    onRideClick?(id: string): void;
}

export const CarDamagesPreview: React.FC<CarDamagesPreviewProps> = function CarDamagesPreview({
    className,
    src,
    title,
    createdAt,
    sessionId,
    onClose,
    onRideClick,
}) {
    const Portal = usePortal();
    const locale = getI18nLocale();

    usePreventScroll({ enabled: true });

    const onRideClickHandler = React.useCallback(() => {
        if (onRideClick && sessionId) {
            onRideClick(sessionId);
        }
    }, [sessionId, onRideClick]);

    return (
        <Portal>
            <div
                className={cn(styles.preview, className)}
                role="dialog"
            >
                <button
                    className={styles.overlay}
                    onClick={onClose}
                />

                <Button
                    className={styles.close}
                    size={ButtonSize.M}
                    icon={CloseIcon}
                    round
                    onClick={onClose}
                />

                <div className={styles.content}>
                    <div className={styles.pictureWrap}>
                        <picture className={styles.picture}>
                            <img
                                className={styles.image}
                                src={src}
                                loading="lazy"
                                alt={title}
                            />
                        </picture>
                    </div>

                    <div className={styles.footer}>
                        {sessionId && (
                            <Button
                                className={styles.rideButton}
                                size={ButtonSize.M}
                                color={ButtonColor.INVERT}
                                label={i18n('Go to the ride')}
                                onClick={onRideClickHandler}
                            />
                        )}

                        <div className={styles.details}>
                            <h5 className={styles.title}>{title}</h5>

                            {createdAt && (
                                <p className={styles.description}>
                                    {i18n('Registered on {date}', {
                                        date: getDateString(createdAt, locale, FORMAT_DATE),
                                    })}
                                </p>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </Portal>
    );
};
