import * as React from 'react';
import cn from 'classnames';

import { EMPTY_DATA } from 'constants/constants';

import { CarBeaconSearchMode } from 'features/CarHardware/ui/CarBeaconSearchMode/CarBeaconSearchMode';

import { CarBeaconSchema } from 'entities/Car/types/CarBeaconSchema';

import { getLastUpdateString } from 'shared/helpers/getLastUpdateString/getLastUpdateString';
import { DetailsContainer, DetailsContainerItemOptions } from 'shared/ui/DetailsContainer/DetailsContainer';
import { GPSValueIcon } from 'shared/ui/GPSValueIcon/GPSValueIcon';

import GSMSignalIcon from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/GSMSignalIcon';

import { i18n } from 'features/CarHardware/ui/CarBeaconHardware/CarBeaconHardware.i18n';

import styles from 'features/CarHardware/ui/CarBeaconHardware/CarBeaconHardware.css';

export interface CarBeaconHardwareProps {
    className?: string;
    carId: string;
    beacon: CarBeaconSchema;
}

export const CarBeaconHardware: React.FC<CarBeaconHardwareProps> = function CarBeaconHardware({
    className,
    carId,
    beacon,
}) {
    // @TODO DRIVEMATICSDEV-355
    const { location, gsm_level, satellites_inview, battery, internal_temperature, imei } = beacon;
    const { longitude, latitude, since, type } = location;
    const updated = since ? getLastUpdateString(since) : EMPTY_DATA;

    const detailsItems: DetailsContainerItemOptions[] = React.useMemo(() => {
        return [
            {
                label: i18n('Updated'),
                id: 'updated',
                value: updated,
            },

            {
                label: 'IMEI',
                id: 'imei',
                value: imei ? `${imei}` : EMPTY_DATA,
            },

            {
                label: i18n('Geodata source'),
                id: 'geosource',
                value: type ? `${type}` : EMPTY_DATA,
            },

            {
                label: i18n('Coordinates'),
                id: 'location',
                value: longitude && latitude ? `${longitude} ${latitude}` : EMPTY_DATA,
            },

            {
                label: i18n('GPS signal'),
                id: 'gps',
                value: satellites_inview !== null ? <GPSValueIcon GPSInview={satellites_inview || 0} /> : EMPTY_DATA,
            },

            {
                label: i18n('GSM signal'),
                id: 'gsm',
                value: gsm_level !== null ? <GSMSignalIcon GSMValue={gsm_level || 0} /> : EMPTY_DATA,
            },

            {
                label: i18n('Temperature'),
                id: 'temperature',
                value: internal_temperature !== null ? `${+internal_temperature.toFixed(1)}°C` : EMPTY_DATA,
            },

            {
                label: i18n('Battery charge'),
                id: 'battery',
                value: battery ? `${battery.toFixed(1)} %` : EMPTY_DATA,
            },
        ];
    }, [battery, gsm_level, imei, internal_temperature, latitude, longitude, satellites_inview, type, updated]);

    return (
        <div className={cn(styles.container, className)}>
            <CarBeaconSearchMode
                carId={carId}
                beacon={beacon}
            />

            <DetailsContainer items={detailsItems} />
        </div>
    );
};
