import * as React from 'react';
import { useContext } from 'react';
import cn from 'classnames/bind';

import { ONE_SECOND } from 'constants/constants';

import { getBeaconSearchMode } from 'utils/getBeaconSearchMode';

import { postCarBeaconSearch } from 'features/CarHardware/api/postCarBeaconSearch/postCarBeaconSearch';

import { isBeaconSearchModeEnabled } from 'entities/Car/helpers/isBeaconSearchModeEnabled/isBeaconSearchModeEnabled';
import { CarBeaconSchema } from 'entities/Car/types/CarBeaconSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { TextWithDotSize } from 'shared/consts/TextWithDotSize';
import { Button } from 'shared/ui/Button/Button';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';

import { i18n } from 'features/CarHardware/ui/CarBeaconSearchMode/CarBeaconSearchMode.i18n';

import styles from 'features/CarHardware/ui/CarBeaconSearchMode/CarBeaconSearchMode.css';

export interface CarBeaconSearchModeProps {
    className?: string;
    carId: string;
    beacon: CarBeaconSchema;
}

const cx = cn.bind(styles);

enum STATUSES {
    ON = 'on',
    OFF = 'off',
    TURN_ON = 'turn_on',
    TURN_OFF = 'turn_off',
}

export const CarBeaconSearchMode: React.FC<CarBeaconSearchModeProps> = function CarBeaconSearchMode({
    className,
    carId,
    beacon,
}) {
    const { beacon_search_state } = beacon;

    const isSwitchBeaconSearchModeEnabled = getBeaconSearchMode();
    const { notificationDispatch } = useContext(NotificationCenterContext) || {};
    const [searchState, setSearchState] = React.useState(beacon_search_state);
    const operationModeIsEnabled = isBeaconSearchModeEnabled(beacon);
    let status: STATUSES;

    if (operationModeIsEnabled) {
        status = searchState ? STATUSES.ON : STATUSES.TURN_OFF;
    } else {
        status = searchState ? STATUSES.TURN_ON : STATUSES.OFF;
    }

    const onSwitchClick = React.useCallback(async () => {
        const newSearchState = !searchState;

        try {
            await postCarBeaconSearch({
                carId,
                state: newSearchState,
                imei: beacon.imei,
            });

            notificationDispatch(
                addNotification({
                    title: newSearchState ? i18n('Search mode was enabled') : i18n('Search mode was disabled'),
                }),
            );

            setSearchState(newSearchState);
        } catch ({ error }) {
            let notification = {
                title: `Error: ${error.error_details?.http_code}`,
                iconType: NotificationIconType.ERROR,
                description: `${error.error_details?.debug_message}`,
                delay: ONE_SECOND,
            };

            notificationDispatch(addNotification(notification));
        }
    }, [beacon.imei, carId, notificationDispatch, searchState]);

    return (
        <div className={cx(styles.container, className)}>
            <div className={styles.textContainer}>
                {isSwitchBeaconSearchModeEnabled && (
                    <Button
                        className={styles.button}
                        color={ButtonColor.SECONDARY}
                        size={ButtonSize.M}
                        onClick={onSwitchClick}
                        label={[STATUSES.ON, STATUSES.TURN_ON].includes(status) ? i18n('Turn off') : i18n('Turn on')}
                    />
                )}

                <TextWithDot
                    className={cx(styles.title, [status])}
                    pulsar={operationModeIsEnabled}
                    size={TextWithDotSize.L}
                >
                    {status === STATUSES.ON && i18n('Search mode was enabled')}
                    {status === STATUSES.TURN_ON && i18n('Search mode will start working soon')}
                    {status === STATUSES.TURN_OFF && i18n('Search mode will stop working soon')}
                    {status === STATUSES.OFF && i18n('Search mode was disabled')}
                </TextWithDot>
            </div>

            <div className={styles.text}>
                {i18n('In search mode the beacon will transmit data every 20 minutes instead of 2 times a day')}
            </div>
        </div>
    );
};
