import * as React from 'react';
import cn from 'classnames';

import { EMPTY_DATA } from 'constants/constants';

import { CarCameraStatus } from 'features/CarCameraStatus';
import { CarCameraHardwareTimeline } from 'features/CarHardware/ui/CarCameraHardwareTimeline/CarCameraHardwareTimeline';

import { CarSignalqStatusType } from 'entities/Car/consts/CarSignalqStatusType';
import { getCarSpeed } from 'entities/Car/helpers/getCarSpeed/getCarSpeed';
import { CarCommonSpeedTraitSchema } from 'entities/Car/types/CarCommonSpeedTraitSchema';
import { CarSchema } from 'entities/Car/types/CarSchema';
import { CarSignalqTraitSchema } from 'entities/Car/types/CarSignalqTraitSchema';

import { DetailsContainer, DetailsContainerItemOptions } from 'shared/ui/DetailsContainer/DetailsContainer';

import { i18n } from 'features/CarHardware/ui/CarCameraHardware/CarCameraHardware.i18n';

import styles from 'features/CarHardware/ui/CarCameraHardware/CarCameraHardware.css';

export interface CarCameraHardwareProps {
    className?: string;
    car: CarSchema & CarCommonSpeedTraitSchema & CarSignalqTraitSchema;
}

const LEGEND: CarSignalqStatusType[] = [
    CarSignalqStatusType.TURNED_ON,
    CarSignalqStatusType.OFFLINE,
    CarSignalqStatusType.TURNED_OFF,
    CarSignalqStatusType.FACED_AWAY,
    CarSignalqStatusType.CAMERA_CLOSED,
];

export const CarCameraHardware: React.FC<CarCameraHardwareProps> = function CarCameraHardware({ className, car }) {
    const { signalq_status, signalq } = car;
    const { serial_number } = signalq;
    const speed = getCarSpeed(car);

    const detailsItems: DetailsContainerItemOptions[] = React.useMemo(() => {
        return [
            {
                label: i18n('Status'),
                id: 'status',
                value: <CarCameraStatus status={signalq_status} />,
            },

            {
                label: i18n('Serial number'),
                id: 'serial',
                value: serial_number ?? EMPTY_DATA,
            },

            {
                label: i18n('Current speed'),
                id: 'speed',
                value: speed ?? EMPTY_DATA,
            },
        ];
    }, [car, serial_number, speed]);

    return (
        <div className={cn(styles.container, className)}>
            <div className={styles.cameraHistory}>
                <div className={styles.timelineHeader}>
                    <div className={styles.timelineTitle}>{i18n('24h camera history')}</div>

                    <div className={styles.tilelineLegend}>
                        {LEGEND.map((status) => (
                            <CarCameraStatus
                                key={status}
                                status={status}
                                shortLabel
                            />
                        ))}
                    </div>
                </div>

                <CarCameraHardwareTimeline
                    className={styles.timeline}
                    carId={car.id}
                />
            </div>

            <DetailsContainer items={detailsItems} />
        </div>
    );
};
