import * as React from 'react';
import cn from 'classnames/bind';

import { ONE_DAY } from 'constants/constants';

import {
    useCarCameraStatusHistoryIntervalsList,
    UseCarCameraStatusHistoryIntervalsListResource,
} from 'features/CarHardware/api/useCarCameraStatusHistoryIntervalsList/useCarCameraStatusHistoryIntervalsList';

import { ErrorBoundary } from 'shared/ui/ErrorBoundary/ErrorBoundary';

import { i18n } from 'features/CarHardware/ui/CarCameraHardwareTimeline/CarCameraHardwareTimeline.i18n';

import styles from 'features/CarHardware/ui/CarCameraHardwareTimeline/CarCameraHardwareTimeline.css';

export interface CarCameraHardwareTimelineProps {
    className?: string;
    carId: string;
}

export interface CarCameraHardwareTimelineContentProps {
    resource: UseCarCameraStatusHistoryIntervalsListResource;
    since: Date;
    until: Date;
}

const cx = cn.bind(styles);

const TEN = 10;

const CarCameraHardwareTimelineContent: React.FC<CarCameraHardwareTimelineContentProps> =
    function CarCameraHardwareTimelineContent({ resource, since, until }) {
        const intervals = resource.read();

        const start = since.getTime();
        const finish = until.getTime();
        const length = finish - start;

        function x(time: Date, fractionDigits: number = 2): number {
            const point = Math.min(finish, Math.max(0, time.getTime() - start));
            const fraction = Math.pow(TEN, fractionDigits);

            return Math.round((point * fraction * 100) / length) / fraction;
        }

        return (
            <ul className={styles.container}>
                {intervals.map(({ status, start_at, end_at }, ind) => {
                    const sx = x(start_at);
                    const fx = x(end_at) - sx;

                    return (
                        <li
                            className={cx(styles.bar, [status])}
                            style={{ left: sx + '%', width: fx + '%' }}
                            key={ind}
                        />
                    );
                })}
            </ul>
        );
    };

export const CarCameraHardwareTimeline: React.FC<CarCameraHardwareTimelineProps> = React.memo(
    function CarCameraHardwareTimeline({ className, carId }) {
        const until = new Date();
        const since = new Date(until.getTime() - ONE_DAY);

        const cameraStatusIntervalsList = useCarCameraStatusHistoryIntervalsList({ carId, since, until });

        return (
            <div className={cx(styles.timeline, className)}>
                <ErrorBoundary
                    fallback={<span className={cx(styles.label, styles.error)}>{i18n('loading error')}</span>}
                >
                    <React.Suspense fallback={<span className={styles.label}>{i18n('loading...')}</span>}>
                        <CarCameraHardwareTimelineContent
                            resource={cameraStatusIntervalsList}
                            since={since}
                            until={until}
                        />
                    </React.Suspense>
                </ErrorBoundary>
            </div>
        );
    },
);
