import * as React from 'react';
import { useHistory, useParams } from 'react-router-dom';
import cn from 'classnames/bind';

import { isCarAddFlag } from 'utils/isCarAddFlag';
import { isCarOverviewEditFlag } from 'utils/isCarOverviewEdit';
import { isTelematicsConfigFlag } from 'utils/isTelematicsConfigFlag';

import { CarCameraStatus } from 'features/CarCameraStatus';
import { CarHardwareTabs } from 'features/CarHardware/consts/CarHardwareTabs';
import { CarBeaconHardware } from 'features/CarHardware/ui/CarBeaconHardware/CarBeaconHardware';
import { CarCameraHardware } from 'features/CarHardware/ui/CarCameraHardware/CarCameraHardware';
import { CarMachineHardware } from 'features/CarHardware/ui/CarMachineHardware/CarMachineHardware';
import { CarTelematicsHardware } from 'features/CarHardware/ui/CarTelematicsHardware/CarTelematicsHardware';
import { CarHardwareInstallButton } from 'features/CarHardwareInstall';
import { CarStatus } from 'features/CarStatus';

import { fetchCarTelematicsConfiguration } from 'entities/Car/api/fetchCarTelematicsConfiguration/fetchCarTelematicsConfiguration';
import { postCarTelematicsConfiguration } from 'entities/Car/api/postCarTelematicsConfiguration/postCarTelematicsConfiguration';
import { UseCarResource } from 'entities/Car/api/useCar/useCar';
import { UseCarBeaconResource } from 'entities/Car/api/useCarBeacon/useCarBeacon';
import { UseCarTelematicsResource } from 'entities/Car/api/useCarTelematics/useCarTelematics';
import { isBeaconSearchModeEnabled } from 'entities/Car/helpers/isBeaconSearchModeEnabled/isBeaconSearchModeEnabled';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Path } from 'shared/consts/Path';
import { TabSize } from 'shared/consts/TabSize';
import { TextWithDotSize } from 'shared/consts/TextWithDotSize';
import { createFile } from 'shared/helpers/createFile/createFile';
import { downloadFile } from 'shared/helpers/downloadFile/downloadFile';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { useUploadFile } from 'shared/hooks/useUploadFile/useUploadFile';
import { Button } from 'shared/ui/Button/Button';
import { Dropdown } from 'shared/ui/Dropdown/Dropdown';
import { Link } from 'shared/ui/Link/Link';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { TabItemOptions } from 'shared/ui/TabItem/TabItem';
import { Tabs } from 'shared/ui/Tabs/Tabs';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';
import { Widget } from 'shared/ui/Widget/Widget';

import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';

import { i18n } from 'features/CarHardware/ui/CarHardware/CarHardware.i18n';

import EditIcon from 'shared/ui/Icons/images/edit-outline-16.inline.svg';
import MoreIcon from 'shared/ui/Icons/images/more-horizontal-24.inline.svg';

import styles from 'features/CarHardware/ui/CarHardware/CarHardware.css';

export interface CarHardwareProps {
    className?: string;

    resource: UseCarResource;
    beaconResource?: UseCarBeaconResource;
    carTelematicsResource?: UseCarTelematicsResource;
    onHardwareChange?(): void;
}

enum UploadFileStatus {
    NONE = 'none',
    LOADING = 'loading',
    SUCCESS = 'success',
    FAILED = 'failed',
}
const UPLOAD_FILE_TEXT: Record<UploadFileStatus, string> = {
    [UploadFileStatus.NONE]: '',
    [UploadFileStatus.LOADING]: i18n('Uploading new config...'),
    [UploadFileStatus.SUCCESS]: i18n('The new config was successfully applied'),
    [UploadFileStatus.FAILED]: i18n('Failed to upload config.'),
};

const cx = cn.bind(styles);

export const CarHardware: React.FC<CarHardwareProps> = function CarHardware({
    className,
    resource,
    beaconResource,
    carTelematicsResource,
    onHardwareChange,
}) {
    // TODO: move to own component params
    const { id } = useParams<{ id: string }>();

    const history = useHistory();
    const params = useParams();
    const car = useResourceReader(resource);

    const { signalq_status, signalq, status, imei } = car || {};
    const beacon = useResourceReader(beaconResource);
    const carTelematics = useResourceReader(carTelematicsResource);

    const { notificationDispatch } = React.useContext(NotificationCenterContext) || {};

    const hasCamera = signalq && signalq_status;
    const hasTelematics = carTelematics && status;

    const isBeaconSearchMode = isBeaconSearchModeEnabled(beacon);

    const items = React.useMemo(
        (): TabItemOptions[] => [
            ...(hasCamera
                ? [
                      {
                          label: (
                              <CarCameraStatus
                                  status={signalq_status}
                                  size={TextWithDotSize.L}
                                  customLabel={i18n('Camera')}
                              />
                          ),

                          value: CarHardwareTabs.CAMERA,
                      },
                  ]
                : []),
            ...(hasTelematics
                ? [
                      {
                          label: (
                              <CarStatus
                                  status={status}
                                  size={TextWithDotSize.L}
                                  customLabel={i18n('Telematics')}
                              />
                          ),

                          value: CarHardwareTabs.TELEMATICS,
                      },
                  ]
                : []),
            ...(beacon
                ? [
                      {
                          label: (
                              <TextWithDot
                                  className={cx({ green: true })}
                                  pulsar={isBeaconSearchMode}
                                  size={TextWithDotSize.L}
                              >
                                  {i18n('Beacon')}
                              </TextWithDot>
                          ),

                          value: CarHardwareTabs.BEACON,
                      },
                  ]
                : []),
            { label: i18n('Car'), value: CarHardwareTabs.CAR },
        ],

        [hasCamera, status, hasTelematics, beacon, isBeaconSearchMode],
    );

    const [currentItem, setCurrentItem] = React.useState<string>(items[0].value);

    const onItemClickHandler = React.useCallback((value: string) => {
        setCurrentItem(value);
    }, []);

    const onEditClick = React.useCallback(() => {
        // @TODO DRIVEMATICSDEV-426
    }, []);

    const onTelematicsDetailsClick = React.useCallback(() => {
        history.push(generateRouterPath(Path.CAR_TELEMATICS, params));
    }, [history, params]);

    const isTelematicsTab = currentItem === CarHardwareTabs.TELEMATICS && carTelematics;
    const isBeaconTab = currentItem === CarHardwareTabs.BEACON && beacon;

    const [uploadStatus, setUploadStatus] = React.useState<UploadFileStatus>(UploadFileStatus.NONE);
    const onUploadFileChange = React.useCallback(
        (file: File) => {
            setUploadStatus(UploadFileStatus.LOADING);

            let reader = new FileReader();

            reader.readAsText(file);

            reader.onload = function () {
                if (typeof reader.result === 'string' && imei) {
                    try {
                        postCarTelematicsConfiguration(imei, JSON.parse(reader.result))
                            .ready()
                            .then(() => {
                                setUploadStatus(UploadFileStatus.SUCCESS);
                            })
                            .catch((e) => {
                                setUploadStatus(UploadFileStatus.FAILED);
                            });
                    } catch (e) {
                        setUploadStatus(UploadFileStatus.FAILED);
                    }
                }
            };

            reader.onerror = function () {
                setUploadStatus(UploadFileStatus.FAILED);
            };
        },
        [imei],
    );

    // TODO rewrite to ButtonFile
    const { upload } = useUploadFile(onUploadFileChange);

    const onUploadConfigClick = React.useCallback(() => {
        upload();
    }, [upload]);
    const onDownloadConfigClick = React.useCallback(() => {
        if (imei) {
            fetchCarTelematicsConfiguration(imei)
                .ready()
                .then((config) => {
                    downloadFile(
                        createFile(JSON.stringify(config), `telematics_config_${Date.now()}.json`, 'text/plain'),
                    );
                })
                .catch(() => {
                    notificationDispatch(
                        addNotification({
                            title: i18n('Failed to download config'),
                            iconType: NotificationIconType.ERROR,
                        }),
                    );
                });
        }
    }, [imei, notificationDispatch]);

    const configMenuItems = React.useMemo((): MenuItemOptions[] => {
        return [
            {
                value: 'upload-config',
                label: i18n('Upload another config'),
                onClick: onUploadConfigClick,
            },
            {
                value: 'download-config',
                label: i18n('Download config'),
                onClick: onDownloadConfigClick,
            },
        ];
    }, [onDownloadConfigClick, onUploadConfigClick]);

    if (!car) {
        return null;
    }

    return (
        <Widget className={cn(styles.container, className)}>
            {items.length === 1 ? (
                <div className={styles.soloTab}>{items[0].label}</div>
            ) : (
                <Tabs
                    className={styles.tabs}
                    items={items}
                    size={TabSize.L}
                    onItemClick={onItemClickHandler}
                    currentItem={currentItem}
                />
            )}

            <div className={styles.content}>
                {currentItem === CarHardwareTabs.CAMERA && signalq && <CarCameraHardware car={car} />}

                {isTelematicsTab && (
                    <CarTelematicsHardware
                        car={car}
                        telematics={carTelematics}
                    />
                )}

                {isBeaconTab && (
                    <CarBeaconHardware
                        carId={id}
                        beacon={beacon}
                    />
                )}

                {currentItem === CarHardwareTabs.CAR && <CarMachineHardware car={car} />}

                <div className={styles.buttons}>
                    {isTelematicsTab && (
                        <Button
                            color={ButtonColor.SECONDARY}
                            size={ButtonSize.M}
                            onClick={onTelematicsDetailsClick}
                            label={i18n('View telematics details')}
                        />
                    )}

                    {isBeaconTab && (
                        <Button
                            color={ButtonColor.SECONDARY}
                            size={ButtonSize.M}
                            onClick={onTelematicsDetailsClick}
                            label={i18n('View beacon details')}
                        />
                    )}

                    {isCarOverviewEditFlag() && (
                        <Button
                            color={ButtonColor.SECONDARY}
                            size={ButtonSize.M}
                            icon={EditIcon}
                            onClick={onEditClick}
                            label={i18n('Edit')}
                        />
                    )}

                    {isCarAddFlag() && (
                        <CarHardwareInstallButton
                            carId={car.id}
                            onChange={onHardwareChange}
                        />
                    )}

                    {isTelematicsConfigFlag() && isTelematicsTab && (
                        <Dropdown
                            size={ButtonSize.M}
                            color={ButtonColor.SECONDARY}
                            className={styles.dropdown_button}
                            icon={MoreIcon}
                            items={configMenuItems}
                            hasArrow={false}
                            position="bottom-end"
                            data-id="config"
                        />
                    )}
                </div>

                {uploadStatus !== UploadFileStatus.NONE && (
                    <div className={styles.uploadConfigStatus}>
                        <span className={cx({}, uploadStatus)}>{UPLOAD_FILE_TEXT[uploadStatus]}</span>
                        {uploadStatus === UploadFileStatus.FAILED && (
                            <Link
                                className={styles.tryAgain}
                                onClick={onUploadConfigClick}
                            >
                                {i18n('Try again')}
                            </Link>
                        )}
                    </div>
                )}
            </div>
        </Widget>
    );
};
