import * as React from 'react';
import cn from 'classnames/bind';

import { EMPTY_DATA } from 'constants/constants';

import { copyToClipboard } from 'utils/copyToClipboard';
import { isCarAlarmSystemFlag } from 'utils/isCarAlarmSystemFlag';
import { isCarTelematicsPasswordFlag } from 'utils/isCarTelematicsPasswordFlag';
import { showCarScoring } from 'utils/showCarScoring';

import { fetchCarTelematicsPassword } from 'features/CarHardware/api/fetchCarTelematicsPassword/fetchCarTelematicsPassword';
import { CarStatus } from 'features/CarStatus/ui/CarStatus/CarStatus';

import { CarTelematicsRes } from 'entities/Car/api/useCarTelematics/useCarTelematics';
import { CarStatusType } from 'entities/Car/consts/CarStatusType';
import { getCarTelematicSpeed } from 'entities/Car/helpers/getCarTelematicSpeed/getCarTelematicSpeed';
import { getCarTelematicsValue } from 'entities/Car/helpers/getCarTelematicsValue/getCarTelematicsValue';
import { CarSchema } from 'entities/Car/types/CarSchema';

import { getLastUpdateString } from 'shared/helpers/getLastUpdateString/getLastUpdateString';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { DetailsContainer, DetailsContainerItemOptions } from 'shared/ui/DetailsContainer/DetailsContainer';
import { GPSValueIcon } from 'shared/ui/GPSValueIcon/GPSValueIcon';
import { Scoring } from 'shared/ui/Scoring/Scoring';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import GSMSignalIcon, { GSMSignalStrength } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/GSMSignalIcon';
import { default as CarStatusComponent } from 'components/Cars/CarStatus';

import { i18n } from 'features/CarHardware/ui/CarTelematicsHardware/CarTelematicsHardware.i18n';

import EyeHideIcon from 'shared/ui/Icons/images/eye-hide-outline-16.inline.svg';
import EyeIcon from 'shared/ui/Icons/images/eye-outline-16.inline.svg';

import styles from 'features/CarHardware/ui/CarTelematicsHardware/CarTelematicsHardware.css';

export interface CarTelematicsHardwareProps {
    className?: string;
    car: CarSchema;
    telematics: CarTelematicsRes;
}

const cx = cn.bind(styles);

export const CarTelematicsHardware: React.FC<CarTelematicsHardwareProps> = function CarTelematicsHardware({
    className,
    car,
    telematics,
}) {
    const { id, status, telematics: carTelematics } = car;
    const { fuel_level: fuelLevel, fuel_distance: fuelDistance } = carTelematics || {};
    const isNoSignalCar = status === CarStatusType.NO_SIGNAL;

    const getValue = getCarTelematicsValue.bind(null, telematics);

    const lastUpdate = telematics?.gsm_signal_level?.updated;
    const lastUpdateText = lastUpdate ? getLastUpdateString(lastUpdate) : EMPTY_DATA;

    const aggressiveRank = car.aggressive_rank;

    const isEngineOn = getValue('engine_on') === 1;
    const speed = getCarTelematicSpeed(telematics);

    const extVoltage = getValue('ext_voltage');

    const engineTemperatureValue = getValue('engine_temperature');
    const engineTemperature = typeof engineTemperatureValue === 'number' && engineTemperatureValue.toFixed(1);
    const engineTemperatureUpdated = telematics.engine_temperature?.updated;

    const gsmSignalLevel = isNoSignalCar ? GSMSignalStrength.ZERO : getValue('gsm_signal_level');
    const GPSInview = isNoSignalCar ? 0 : getValue('VEGA_GPS_INVIEW');
    const GPSUsed = isNoSignalCar ? 0 : getValue('gps_used');
    const doorLock = getValue('can_cls_state');
    const doorLockSupported = typeof doorLock === 'number' && (doorLock === 1 || doorLock === 0);

    const alarmSystem = Boolean(getValue('2856'));
    const alertFired = Boolean(getValue('2857'));

    const alert = alertFired
        ? { color: styles.red, label: i18n('Alarm fired') }
        : alarmSystem
        ? {
              color: styles.green,
              label: i18n('On'),
          }
        : { color: styles.gray, label: i18n('Off') };

    const [pass, setPass] = React.useState<Optional<string>>(undefined);
    const [passHide, setPassHide] = React.useState<boolean>(true);
    const onPasswordCopy = React.useCallback(() => {
        if (pass) {
            copyToClipboard(pass);
        }
    }, [pass]);

    const cacheContext = useCacheRequestContext();
    const onPasswordClick = React.useCallback(() => {
        if (!pass) {
            const request = fetchCarTelematicsPassword({ carId: id, cache: cacheContext });

            request.then((result) => {
                setPass(result.password);
                setPassHide(false);
            });
        } else {
            setPassHide(!passHide);
        }
    }, [cacheContext, id, pass, passHide]);

    // @TODO DRIVEMATICSDEV-354
    const detailsItems: DetailsContainerItemOptions[] = React.useMemo(() => {
        return [
            {
                label: i18n('Updated'),
                id: 'updated',
                value: lastUpdateText,
            },

            {
                label: i18n('Status'),
                id: 'status',
                value: (
                    <CarStatusComponent
                        carInfo={car}
                        customStatus={<CarStatus status={status} />}
                    />
                ),
            },

            ...(isCarAlarmSystemFlag()
                ? [
                      {
                          label: i18n('Alarm system'),
                          id: 'alarm',
                          value: <TextWithDot className={alert.color}>{alert.label}</TextWithDot>,
                      },
                  ]
                : []),
            {
                label: i18n('Central door lock'),
                id: 'doorLock',
                value: doorLockSupported ? (Boolean(doorLock) ? i18n('Closed') : i18n('Open')) : EMPTY_DATA,
            },
            ...(showCarScoring()
                ? [
                      {
                          label: i18n('Scoring'),
                          id: 'scoring',
                          value: <Scoring value={aggressiveRank} />,
                      },
                  ]
                : []),
            {
                label: i18n('Current speed'),
                id: 'speed',
                value: speed,
            },

            {
                label: i18n('Fuel'),
                id: 'fuelLevel',
                value: fuelLevel && typeof fuelLevel === 'number' ? `${fuelLevel.toFixed(1)} %` : EMPTY_DATA,
            },

            {
                label: i18n('Range'),
                id: 'fuelDistance',
                value: fuelDistance ? `${fuelDistance} ${i18n('km')}` : EMPTY_DATA,
            },

            {
                label: i18n('Battery charge'),
                id: 'extVoltage',
                value: extVoltage && typeof extVoltage === 'number' ? `${extVoltage.toFixed(1)} V` : EMPTY_DATA,
            },

            {
                label: i18n('Engine temp.'),
                id: 'engineTemperature',
                value: engineTemperature ? (
                    car.status &&
                    car.status !== CarStatusType.ENGINE_ON &&
                    car.status !== CarStatusType.RIDING &&
                    engineTemperatureUpdated ? (
                        <div className={styles.temp_engine_off}>
                            {`${engineTemperature} °C` + ` (${getLastUpdateString(engineTemperatureUpdated)})`}
                        </div>
                    ) : (
                        `${engineTemperature} °C`
                    )
                ) : (
                    EMPTY_DATA
                ),
            },

            {
                label: i18n('GPS signal'),
                id: 'gps',
                value: (
                    <GPSValueIcon
                        GPSInview={GPSInview}
                        GPSUsed={GPSUsed}
                    />
                ),
            },

            {
                label: i18n('GSM signal'),
                id: 'gsmSignalLevel',
                value:
                    typeof gsmSignalLevel === 'number' ? (
                        <div>
                            <GSMSignalIcon GSMValue={gsmSignalLevel} />
                        </div>
                    ) : (
                        EMPTY_DATA
                    ),
            },

            ...(isCarTelematicsPasswordFlag()
                ? [
                      {
                          label: i18n('Password'),
                          id: 'password',
                          value: (
                              <span className={styles.password}>
                                  {passHide ? (
                                      <span
                                          onClick={onPasswordClick}
                                          data-testid="show-password"
                                      >
                                          ******
                                          <EyeIcon className={styles.eye} />
                                      </span>
                                  ) : (
                                      <>
                                          <span onClick={onPasswordCopy}>{pass}</span>
                                          <EyeHideIcon
                                              className={styles.eye}
                                              onClick={onPasswordClick}
                                              data-testid="hide-password"
                                          />
                                      </>
                                  )}
                              </span>
                          ),
                      },
                  ]
                : []),
        ];
    }, [
        lastUpdateText,
        car,
        status,
        alert.color,
        alert.label,
        aggressiveRank,
        speed,
        isEngineOn,
        fuelLevel,
        fuelDistance,
        extVoltage,
        engineTemperature,
        engineTemperatureUpdated,
        GPSInview,
        GPSUsed,
        gsmSignalLevel,
        passHide,
        onPasswordClick,
        onPasswordCopy,
        pass,
    ]);

    return (
        <div className={cx(styles.container, [className])}>
            <DetailsContainer
                className={status === CarStatusType.NO_SIGNAL ? styles.nosignal : undefined}
                items={detailsItems}
            />
        </div>
    );
};
