import * as React from 'react';
import cn from 'classnames';

import { CarHardwareBeaconForm } from 'features/CarHardwareInstall/ui/CarHardwareBeaconForm/CarHardwareBeaconForm';
import { CarHardwareCameraForm } from 'features/CarHardwareInstall/ui/CarHardwareCameraForm/CarHardwareCameraForm';
import { CarHardwareTelematicsForm } from 'features/CarHardwareInstall/ui/CarHardwareTelematicsForm/CarHardwareTelematicsForm';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Dropdown } from 'shared/ui/Dropdown/Dropdown';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';

import { i18n } from 'features/CarHardwareInstall/ui/CarHardwareInstall/CarHardwareInstall.i18n';

import LinkIcon from 'shared/ui/Icons/images/link-16.inline.svg';

import styles from 'features/CarHardwareInstall/ui/CarHardwareInstall/CarHardwareInstall.css';

export interface CarHardwareInstallProps {
    className?: string;
}

type CarHardwareType = 'camera' | 'telematics' | 'beacon';

export const CarHardwareInstall: React.FC<CarHardwareInstallProps> = function CarHardwareInstall({ className }) {
    const [state, setState] = React.useState<CarHardwareType[]>([]);

    const items = React.useMemo<MenuItemOptions<CarHardwareType>[]>(() => {
        const items: MenuItemOptions<CarHardwareType>[] = [
            {
                label: i18n('Camera'),
                value: 'camera',
            },

            {
                label: i18n('Telematics'),
                value: 'telematics',
            },

            {
                label: i18n('Beacon'),
                value: 'beacon',
            },
        ];

        return items.filter(({ value }) => !state.includes(value));
    }, [state]);

    const onMenuItemClick = React.useCallback((event: React.MouseEvent<HTMLLIElement>) => {
        const value = event.currentTarget.getAttribute('data-value') as CarHardwareType;

        if (value) {
            setState((state) => [...state, value]);
        }
    }, []);

    return (
        <div className={cn(styles.container, className)}>
            {state.map((type) => {
                if (type === 'camera') {
                    return <CarHardwareCameraForm key={type} />;
                }

                if (type === 'telematics') {
                    return <CarHardwareTelematicsForm key={type} />;
                }

                if (type === 'beacon') {
                    return <CarHardwareBeaconForm key={type} />;
                }

                return null;
            })}

            {Boolean(items.length) && (
                <div>
                    <Dropdown
                        color={ButtonColor.SECONDARY}
                        size={ButtonSize.M}
                        items={items}
                        icon={LinkIcon}
                        hasArrow={false}
                        label={state.length ? i18n('Install another device') : i18n('Install device')}
                        onMenuClick={onMenuItemClick}
                        equalWidth
                    />
                </div>
            )}
        </div>
    );
};
