import * as React from 'react';

import { CarHardwareBeaconModal } from 'features/CarHardwareInstall/ui/CarHardwareBeaconModal/CarHardwareBeaconModal';
import { CarHardwareCameraModal } from 'features/CarHardwareInstall/ui/CarHardwareCameraModal/CarHardwareCameraModal';
import { CarHardwareTelematicsModal } from 'features/CarHardwareInstall/ui/CarHardwareTelematicsModal/CarHardwareTelematicsModal';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Dropdown } from 'shared/ui/Dropdown/Dropdown';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';

import { i18n } from 'features/CarHardwareInstall/ui/CarHardwareInstallButton/CarHardwareInstallButton.i18n';

import LinkIcon from 'shared/ui/Icons/images/link-16.inline.svg';

export interface CarHardwareInstallButtonProps {
    carId: string;
    className?: string;
    onChange?(): void;
}

type CarHardwareType = 'camera' | 'telematics' | 'beacon';

export const CarHardwareInstallButton: React.FC<CarHardwareInstallButtonProps> = function CarHardwareInstallButton({
    carId,
    className,
    onChange,
}) {
    const [state, setState] = React.useState<CarHardwareType>();

    const items = React.useMemo<MenuItemOptions<CarHardwareType>[]>(() => {
        return [
            {
                label: i18n('Camera'),
                value: 'camera',
            },

            {
                label: i18n('Telematics'),
                value: 'telematics',
            },

            {
                label: i18n('Beacon'),
                value: 'beacon',
            },
        ];
    }, []);

    const onMenuItemClick = React.useCallback((event: React.MouseEvent<HTMLLIElement>) => {
        const value = event.currentTarget.getAttribute('data-value');

        if (value) {
            setState(value as CarHardwareType);
        }
    }, []);

    const onModalClose = React.useCallback(() => {
        setState(undefined);
    }, []);

    return (
        <>
            <Dropdown
                className={className}
                color={ButtonColor.PRIMARY}
                size={ButtonSize.M}
                items={items}
                icon={LinkIcon}
                hasArrow={false}
                label={i18n('Install device')}
                onMenuClick={onMenuItemClick}
                equalWidth
            />

            {state === 'camera' && (
                <CarHardwareCameraModal
                    carId={carId}
                    onClose={onModalClose}
                    onChange={onChange}
                />
            )}

            {state === 'telematics' && (
                <CarHardwareTelematicsModal
                    carId={carId}
                    onClose={onModalClose}
                    onChange={onChange}
                />
            )}

            {state === 'beacon' && (
                <CarHardwareBeaconModal
                    carId={carId}
                    onClose={onModalClose}
                    onChange={onChange}
                />
            )}
        </>
    );
};
