import * as React from 'react';

import { CarHardwareTelematicsFormSchema } from 'features/CarHardwareInstall/types/CarHardwareTelematicsFormSchema';

import { InputSize } from 'shared/consts/InputSize';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { Select } from 'shared/ui/Select/Select';

import { i18n } from 'features/CarHardwareInstall/ui/CarHardwareTelematicsForm/CarHardwareTelematicsForm.i18n';

export interface CarHardwareTelematicsFormProps {
    className?: string;

    controller?: UseFormControllerSub<OptionalRecord<CarHardwareTelematicsFormSchema>>;
}

const VALIDATION_RULES: UseFormValidation<OptionalRecord<CarHardwareTelematicsFormSchema>> = {
    modelType: validateRequired(),
    imei: validateRequired(),
};

const INIT_FORM: OptionalRecord<CarHardwareTelematicsFormSchema> = {
    modelType: 'MT-32K LTE',
    imei: '',
};

export const CarHardwareTelematicsForm: React.FC<CarHardwareTelematicsFormProps> = function CarHardwareTelematicsForm({
    className,
    controller,
}) {
    const { update, watch, errors } = useForm<OptionalRecord<CarHardwareTelematicsFormSchema>>({
        init: { ...INIT_FORM },
        validation: VALIDATION_RULES,
        controller,
    });

    const telematicsTypeItems = React.useMemo<MenuItemOptions[]>(() => {
        return [
            {
                label: 'Drivematics Vega MT-32',
                value: 'MT-32K LTE',
            },
        ];
    }, []);

    return (
        <FormContainer className={className}>
            <FormLabel title={i18n('Type')}>
                <FormErrorContainer error={<ErrorLabel errors={errors.modelType} />}>
                    <Select
                        items={telematicsTypeItems}
                        checkedMenu={watch('modelType')}
                        onSelectChange={update('modelType') as any}
                        hasError={Boolean(errors.modelType)}
                    />
                </FormErrorContainer>
            </FormLabel>

            <FormLabel title={i18n('IMEI')}>
                <FormErrorContainer error={<ErrorLabel errors={errors.imei} />}>
                    <Input
                        inputSize={InputSize.M}
                        placeholder={i18n('15-digit number on the back of the telematics block')}
                        hasClear
                        onInputChange={update('imei')}
                        hasError={Boolean(errors.imei)}
                    />
                </FormErrorContainer>
            </FormLabel>

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
