import * as React from 'react';
import { useHistory, useParams } from 'react-router-dom';

import getSessionId from 'utils/sessions/getSessionId';

import { CarLatestRideRow } from 'features/CarLatestRide/ui/CarLatestRideRow/CarLatestRideRow';

import { UseCarRidesResource } from 'entities/Car/api/useCarRides/useCarRides';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Path } from 'shared/consts/Path';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { Button } from 'shared/ui/Button/Button';
import { Island } from 'shared/ui/Island/Island';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/CarLatestRide/ui/CarLatestRide/CarLatestRide.i18n';

import styles from 'features/CarLatestRide/ui/CarLatestRide/CarLatestRide.css';

export interface CarLatestRideProps {
    resource: UseCarRidesResource;
}

const MAX_TABLE_ROWS_LENGTH = 5;

export const CarLatestRide: React.FC<CarLatestRideProps> = function CarLatestRide({ resource }) {
    const history = useHistory();
    const params = useParams();

    const resourceData = useResourceReader(resource);
    const { rides = [] } = resourceData || {};
    const hasRides = rides.length > 0;
    const showMoreButton = rides.length > MAX_TABLE_ROWS_LENGTH;

    const data = rides.slice(0, MAX_TABLE_ROWS_LENGTH);

    const onMoreClick = React.useCallback(() => {
        history.push(generateRouterPath(Path.CAR_RIDES, params));
    }, [history, params]);

    return (
        <Island title={i18n('Latest rides')}>
            {hasRides ? (
                <div className={styles.tableContainer}>
                    <Table className={styles.table}>
                        <Table.Head>
                            <Table.Row>
                                <Table.Th gap />
                                <Table.Th>{i18n('Date and time')}</Table.Th>
                                <Table.Th>{i18n('Duration')}</Table.Th>
                                <Table.Th>{i18n('Mileage')}</Table.Th>
                                <Table.Th gap />
                            </Table.Row>
                        </Table.Head>
                        <Table.Body>
                            {data.map((ride, index) => {
                                const sessionId = getSessionId(ride);

                                return (
                                    <CarLatestRideRow
                                        key={sessionId ?? index}
                                        className={styles.row}
                                        ride={ride}
                                    />
                                );
                            })}
                        </Table.Body>
                    </Table>
                </div>
            ) : (
                <div className={styles.noData}>{i18n('There is no data')}</div>
            )}

            {showMoreButton && (
                <Button
                    className={styles.button}
                    color={ButtonColor.SECONDARY}
                    size={ButtonSize.M}
                    onClick={onMoreClick}
                    label={i18n('View all rides')}
                />
            )}
        </Island>
    );
};
