import * as React from 'react';

import { checkModelForBlock } from 'utils/checkModelForBlock';
import { isManageDoors } from 'utils/isManageDoors';
import { isShowCarService } from 'utils/isShowCarService';

import { CarLocationAreaStatus } from 'features/CarLocation/ui/CarLocationAreaStatus/CarLocationAreaStatus';
import { CarLocationButtons } from 'features/CarLocation/ui/CarLocationButtons/CarLocationButtons';
import { CarLocationMap } from 'features/CarLocation/ui/CarLocationMap/CarLocationMap';

import { UseCarResource } from 'entities/Car/api/useCar/useCar';
import { UseCarBeaconResource } from 'entities/Car/api/useCarBeacon/useCarBeacon';
import { CarLocationTags } from 'entities/Car/consts/CarLocationTags';
import { getCarSignalqStatus } from 'entities/Car/helpers/getCarSignalqStatus/getCarSignalqStatus';
import { getCarStatus } from 'entities/Car/helpers/getCarStatus/getCarStatus';

import { getLastUpdateString } from 'shared/helpers/getLastUpdateString/getLastUpdateString';
import { useLocationAddress } from 'shared/hooks/useLocationAddress/useLocationAddress';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { DotText } from 'shared/ui/DotText/DotText';
import { Widget } from 'shared/ui/Widget/Widget';

import styles from 'features/CarLocation/ui/CarLocation/CarLocation.css';

export interface CarLocationProps {
    resource: UseCarResource;
    beaconResource: UseCarBeaconResource;
}

const DENY_AREA_TAG = 'deny';

export const CarLocation: React.FC<CarLocationProps> = function CarLocation({ resource, beaconResource }) {
    const car = useResourceReader(resource);
    const beaconInfo = useResourceReader(beaconResource);

    const { status, location, signalq, signalq_status, model_id } = car || {};
    const { timestamp, lon = null, lat = null, tags: tagsLocation = [] } = location ?? {};
    // @TODO никогда не бывает UNKNOWN
    const area = tagsLocation.includes(DENY_AREA_TAG) ? CarLocationTags.RESTRICTED : CarLocationTags.AVAILABLE;

    const [address, setAddress] = React.useState<string | null>(null);

    const isModelForBlock = model_id && checkModelForBlock(model_id);
    const hasButtons = isManageDoors() || isShowCarService() || isModelForBlock;

    useLocationAddress(lat, lon, setAddress);

    if (!car) {
        return null;
    }

    return (
        <Widget className={styles.container}>
            <h3 className={styles.title}>
                {Boolean(signalq) ? getCarSignalqStatus(signalq_status) : getCarStatus(status)}
            </h3>

            <CarLocationAreaStatus status={area} />

            {beaconInfo ? (
                <CarLocationMap
                    car={car}
                    beaconInfo={beaconInfo}
                />
            ) : (
                <CarLocationMap car={car} />
            )}

            {address && (
                <div
                    key={timestamp}
                    className={styles.location}
                    data-testid="address"
                >
                    <DotText items={[getLastUpdateString(timestamp), address]} />
                </div>
            )}

            {hasButtons && !signalq ? (
                <div className={styles.buttons_container}>
                    <CarLocationButtons resource={resource} />
                </div>
            ) : null}

            {/*@TODO DRIVEMATICSDEV-424*/}
            {/*{isServiceConfirmOpen && <CarLocationServiceDialog resource={resource} onClose={onServiceDialogClose} />}*/}
        </Widget>
    );
};
