import * as React from 'react';
import { useMemo, useState } from 'react';

import { ONE_SECOND } from 'constants/constants';

import { checkModelForBlock } from 'utils/checkModelForBlock';
import { getFlagCarPanicButtons } from 'utils/getFlagCarPanicButtons';
import { isManageDoors } from 'utils/isManageDoors';
import { isShowCarService } from 'utils/isShowCarService';

import { MAX_BUTTONS_COUNT, SERVICE_TAGS } from 'features/CarLocation/consts/constants';

import { ControlType } from 'entities/Car';
import { UseCarResource } from 'entities/Car/api/useCar/useCar';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { useTimerRef } from 'shared/hooks/useTimerRef/useTimerRef';
import { Button } from 'shared/ui/Button/Button';
import { Dropdown } from 'shared/ui/Dropdown/Dropdown';

import CarControlConfirm from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControlConfirm';
import { CAR_BLOCK_TAGS } from 'components/Cars/CarCard/CarCardOverview/CarMainWidget/CarControlConfirm/types';
import { SERVICE_TAG } from 'components/Cars/constants';

import { i18n } from 'features/CarLocation/ui/CarLocationButtons/CarLocationButtons.i18n';

import BanIcon from 'shared/ui/Icons/images/ban-16.inline.svg';
import LockIcon from 'shared/ui/Icons/images/lock-16.inline.svg';
import ShowMoreIcon from 'shared/ui/Icons/images/show-more-16.inline.svg';
import UnlockIcon from 'shared/ui/Icons/images/unlock-16.inline.svg';

import styles from 'features/CarLocation/ui/CarLocationButtons/CarLocationButtons.css';

export interface CarLocationButtonsProps {
    resource: UseCarResource;
}

interface CarButton {
    icon?: SvgIcon | undefined;
    label: string;
    onClick: () => void;
    color?: ButtonColor;
    displayCondition: boolean;
}

export const CarLocationButtons: React.FC<CarLocationButtonsProps> = function CarLocationButtons({ resource }) {
    const car = useResourceReader(resource);
    const timerRef = useTimerRef(0);
    const { model_id, tags } = car || {};
    const isCarInService = tags?.find((tag) => tag.tag === SERVICE_TAG);
    const isModelForBlock = Boolean(model_id && checkModelForBlock(model_id));

    const isCarInBlockProcess = tags?.some((tag) =>
        [
            CAR_BLOCK_TAGS.telematics_deferred_polite_wireless_block,
            CAR_BLOCK_TAGS.telematics_deferred_wireless_block,
        ].includes(tag.tag),
    );

    const isCarBlocked = tags?.some((tag) => tag.tag === CAR_BLOCK_TAGS.leasing_car_is_blocked);
    const isBlocked = isCarInBlockProcess || isCarBlocked;

    const [command, setCommand] = useState<ControlType | null>(null);
    const [isServiceConfirmOpen, setIsServiceConfirmOpen] = useState<boolean>(false);
    const [isBlockConfirmOpen, setIsBlockConfirmOpen] = useState<boolean>(false);

    const onLockClick = React.useCallback(() => {
        setCommand(ControlType.close_doors);
    }, []);

    const onUnlockClick = React.useCallback(() => {
        setCommand(ControlType.open_doors);
    }, []);

    const onClose = React.useCallback(() => {
        setCommand(null);
    }, []);

    const onMoveToServiceClick = React.useCallback(() => {
        setIsServiceConfirmOpen(true);
    }, []);

    const onServiceDialogClose = React.useCallback(() => {
        setIsServiceConfirmOpen(false);

        // @todo: status not actual, waiting 1s
        timerRef.current = setTimeout(() => {
            resource.reload();
        }, ONE_SECOND) as unknown as number;
    }, [resource, timerRef]);

    const onBanClick = React.useCallback(() => {
        setIsBlockConfirmOpen(true);
    }, []);

    const onBeepClick = React.useCallback(() => {
        setCommand(ControlType.beep);
    }, []);

    const onBlockDialogClose = React.useCallback(() => {
        setIsBlockConfirmOpen(false);

        // @todo: status not actual, waiting 1s
        timerRef.current = setTimeout(() => {
            resource.reload();
        }, ONE_SECOND) as unknown as number;
    }, [resource, timerRef]);

    const MANAGE_DOORS = Boolean(isManageDoors());
    const SHOW_CAR_SERVICE = Boolean(isShowCarService());
    const CAR_PANIC_BUTTON = Boolean(getFlagCarPanicButtons());

    const buttons: CarButton[] = useMemo(() => {
        return [
            {
                icon: LockIcon,
                label: i18n('Lock'),
                onClick: onLockClick,
                displayCondition: MANAGE_DOORS,
            },

            {
                icon: UnlockIcon,
                label: i18n('Unlock'),
                onClick: onUnlockClick,
                displayCondition: MANAGE_DOORS,
            },

            {
                label: isCarInService ? i18n('Return from service') : i18n('Move to service'),
                onClick: onMoveToServiceClick,
                displayCondition: SHOW_CAR_SERVICE,
            },

            {
                icon: BanIcon,
                label: isBlocked ? i18n('Unblock car') : i18n('Block car'),
                onClick: onBanClick,
                color: ButtonColor.ALARM,
                displayCondition: isModelForBlock,
            },

            {
                label: i18n('Horn'),
                onClick: onBeepClick,
                displayCondition: CAR_PANIC_BUTTON,
            },
        ].filter((button) => button.displayCondition);
    }, [MANAGE_DOORS, SHOW_CAR_SERVICE, isModelForBlock, CAR_PANIC_BUTTON, isBlocked, isCarInService]);

    const displayButtons = useMemo(() => {
        return buttons.slice(0, MAX_BUTTONS_COUNT);
    }, [buttons]);

    const dropdownButtons = useMemo(() => {
        return buttons.slice(MAX_BUTTONS_COUNT).map((el) => {
            return {
                value: el.label,
                label: el.label,
                onClick: el.onClick,
            };
        });
    }, [buttons]);

    const tagForBlock = isCarBlocked
        ? CAR_BLOCK_TAGS.telematics_deferred_wireless_unblock
        : !isCarInBlockProcess
        ? CAR_BLOCK_TAGS.telematics_deferred_polite_wireless_block
        : null;

    if (!car) {
        return null;
    }

    return (
        <div className={styles.buttons}>
            {buttons.length
                ? displayButtons.map((el) => {
                      return (
                          <Button
                              key={el.label}
                              label={el.label}
                              icon={el.icon}
                              onClick={el.onClick}
                              size={ButtonSize.M}
                              color={el.color ?? ButtonColor.SECONDARY}
                          />
                      );
                  })
                : null}

            {buttons.length > MAX_BUTTONS_COUNT ? (
                <Dropdown
                    items={dropdownButtons}
                    hasArrow={false}
                    opened={true}
                    icon={ShowMoreIcon}
                    color={ButtonColor.SECONDARY}
                    className={styles.dropdownButton}
                    size={ButtonSize.M}
                    data-testid="dropdown"
                />
            ) : null}

            {command && (
                <CarControlConfirm
                    carInfo={car}
                    actionType="command"
                    command={command}
                    onClose={onClose}
                />
            )}

            {isServiceConfirmOpen && (
                <CarControlConfirm
                    title={isCarInService ? i18n('Return from service?') : i18n('Move to service?')}
                    carInfo={car}
                    actionType={isCarInService ? 'remove_tag' : 'add_tag'}
                    addingTag={!isCarInService ? SERVICE_TAG : null}
                    removingTagNames={isCarInService ? SERVICE_TAGS : null}
                    onClose={onServiceDialogClose}
                />
            )}

            {isBlockConfirmOpen && (
                <CarControlConfirm
                    carInfo={car}
                    actionType={isCarInBlockProcess ? 'remove_tag' : 'add_tag'}
                    addingTag={tagForBlock}
                    removingTagNames={
                        isCarInBlockProcess ? [CAR_BLOCK_TAGS.telematics_deferred_polite_wireless_block] : null
                    }
                    onClose={onBlockDialogClose}
                />
            )}
        </div>
    );
};
