import * as React from 'react';
import { useCallback, useState } from 'react';

import { postCarMaintenanceInfo } from 'features/CarMaintenance/api/postCarMaintenanceInfo/postCarMaintenanceInfo';
import { useCarMaintenanceInfo } from 'features/CarMaintenance/api/useCarMaintenanceInfo/useCarMaintenanceInfo';
import {
    CarMaintenanceAddModal,
    CarMaintenanceAddModalForm,
} from 'features/CarMaintenance/ui/CarMaintenanceAddModal/CarMaintenanceAddModal';
import { CarMaintenanceList } from 'features/CarMaintenance/ui/CarMaintenanceList/CarMaintenanceList';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Button } from 'shared/ui/Button/Button';
import { ErrorBoundary } from 'shared/ui/ErrorBoundary/ErrorBoundary';
import { Spin } from 'shared/ui/Spin/Spin';

import { i18n } from 'features/CarMaintenance/ui/CarMaintenance/CarMaintenance.i18n';

import styles from 'features/CarMaintenance/ui/CarMaintenance/CarMaintenance.css';

export interface CarMaintenanceProps {
    vin: string;
}

export const CarMaintenance: React.FC<CarMaintenanceProps> = React.memo(function CarMaintenance({
    vin,
}: CarMaintenanceProps) {
    const maintenanceResource = useCarMaintenanceInfo(vin);

    const [isModalOpen, setModalOpen] = useState<boolean>(false);

    const onModalOpen = useCallback(() => {
        setModalOpen(true);
    }, []);

    const onModalClose = useCallback(() => {
        setModalOpen(false);
    }, []);

    const onFormSubmit = useCallback(
        async ({ date, mileage }: CarMaintenanceAddModalForm) => {
            try {
                await postCarMaintenanceInfo({ vin, date, mileage: Number(mileage) });
                maintenanceResource.reload();
                onModalClose();
            } catch (error) {
                // TODO: return setError('_serverError', getFetchErrorMessage(error));
            }
        },
        [vin],
    );

    return (
        <>
            <h3 className={styles.title}>{i18n('Latest maintenance')}</h3>

            <Button
                className={styles.buttonAdd}
                color={ButtonColor.PRIMARY}
                size={ButtonSize.M}
                label={i18n('Add maintenance')}
                onClick={onModalOpen}
            />

            {/*TODO: Error message*/}
            <ErrorBoundary fallback={<div>Error :(</div>}>
                {/*TODO: replace Spin*/}
                <React.Suspense
                    fallback={
                        <div className={styles.loading}>
                            <Spin className={styles.spin} />
                        </div>
                    }
                >
                    <CarMaintenanceList resource={maintenanceResource} />
                </React.Suspense>
            </ErrorBoundary>

            {isModalOpen && (
                <CarMaintenanceAddModal
                    onClose={onModalClose}
                    onFormSubmit={onFormSubmit}
                />
            )}
        </>
    );
});
