import * as React from 'react';

import { deleteCarService } from 'features/CarService/api/deleteCarService/deleteCarService';
import { postEditCarService } from 'features/CarService/api/postEditCarService/postEditCarService';
import { CarDeleteServiceDialog } from 'features/CarService/ui/CarDeleteServiceDialog/CarDeleteServiceDialog';
import { CarServiceForm } from 'features/CarService/ui/CarServiceForm/CarServiceForm';

import { useCarService } from 'entities/Car/api/useCarService/useCarService';
import { CarSearchSchema } from 'entities/Car/types/CarSearchSchema';
import { CarServiceFormSchema } from 'entities/Car/types/CarServiceFormSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/CarService/ui/CarEditServiceModal/CarEditServiceModal.i18n';

export interface CarEditServiceModalProps extends Pick<ModalContainerProps, 'onClose'> {
    tagId: string;
    car?: CarSearchSchema;

    onChange?(): void;

    onClose?(): void;
}

export const CarEditServiceModal: React.FC<CarEditServiceModalProps> = function CarEditServiceModal({
    tagId,
    car,
    onChange,
    onClose,
}) {
    const { data, isLoading } = useCarService(tagId);

    const initial = { ...data, car };

    const [isDeleteDialogOpen, setDeleteDialogOpen] = React.useState<boolean>(false);

    const { isEqualData, onFormChangeHandler } = useEqualDataForm<CarServiceFormSchema>(initial);
    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<CarServiceFormSchema>>();

    const onDeleteDialogOpen = React.useCallback(() => {
        setDeleteDialogOpen(true);
    }, []);

    const onDeleteDialogClose = React.useCallback(() => {
        setDeleteDialogOpen(false);
    }, []);

    const onSubmitHandler = React.useCallback(async () => {
        if (validate()) {
            const data = getValues();

            try {
                await postEditCarService(data as CarServiceFormSchema, { tagId });

                // @todo: add notification
            } catch (error) {
                return setError('_serverError', getFetchErrorMessage(error));
            }

            if (onChange) {
                onChange();
            }

            if (onClose) {
                onClose();
            }
        }
    }, [tagId, onChange, onClose, getValues, validate]);

    const onDeleteSubmitHandler = React.useCallback(async () => {
        try {
            await deleteCarService(tagId);

            // @todo: add notification
        } catch (error) {
            onDeleteDialogClose();

            return setError('_serverError', getFetchErrorMessage(error));
        }

        onDeleteDialogClose();

        if (onChange) {
            onChange();
        }

        if (onClose) {
            onClose();
        }
    }, [tagId, onChange, onClose, onDeleteDialogClose]);

    if (!data || isLoading) {
        return null;
    }

    return (
        <>
            <ModalContainer
                title={i18n('Edit service')}
                hasClose
                controls={
                    <>
                        <Button
                            color={ButtonColor.PRIMARY}
                            size={ButtonSize.M}
                            label={i18n('Save')}
                            disabled={isEqualData}
                            onClick={onSubmitHandler}
                        />

                        <Button
                            color={ButtonColor.ALARM}
                            size={ButtonSize.M}
                            label={i18n('Delete')}
                            onClick={onDeleteDialogOpen}
                        />
                    </>
                }
                onClose={onClose}
            >
                <CarServiceForm
                    initial={initial}
                    controller={controller}
                    disableCarSwitch
                    onFormChange={onFormChangeHandler}
                />
            </ModalContainer>

            {isDeleteDialogOpen && (
                <CarDeleteServiceDialog
                    onDialogSubmit={onDeleteSubmitHandler}
                    onClose={onDeleteDialogClose}
                />
            )}
        </>
    );
};
