import * as React from 'react';
import cn from 'classnames/bind';

import { DEFAULT_SERVICE_TIME } from 'features/CarService/consts/constants';

import { CAR_SERVICE_MENU_ITEMS, CarInput } from 'entities/Car';
import { CarServiceJobType } from 'entities/Car/consts/CarServiceJobType';
import { CarServiceFormSchema } from 'entities/Car/types/CarServiceFormSchema';

import { InputSize } from 'shared/consts/InputSize';
import { TextareaSize } from 'shared/consts/TextareaSize';
import { TooltipTailPosition } from 'shared/consts/TooltipTailPosition';
import { TooltipTailSide } from 'shared/consts/TooltipTailSide';
import { formatTime } from 'shared/helpers/formatTime/formatTime';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { validateNumber } from 'shared/helpers/validateNumber/validateNumber';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormDateTimeRange } from 'shared/ui/FormDateTimeRange/FormDateTimeRange';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { Link } from 'shared/ui/Link/Link';
import { Select } from 'shared/ui/Select/Select';
import { Textarea } from 'shared/ui/Textarea/Textarea';
import { Tooltip } from 'shared/ui/Tooltip/Tooltip';

import { i18n } from 'features/CarService/ui/CarServiceForm/CarServiceForm.i18n';

import styles from 'features/CarService/ui/CarServiceForm/CarServiceForm.css';

export interface CarServiceFormProps {
    className?: string;

    initial?: Partial<CarServiceFormSchema>;
    controller?: UseFormControllerSub<CarServiceFormSchema>;

    disableCarSwitch?: boolean;

    onFormChange?(state: CarServiceFormSchema): void;
}

const cx = cn.bind(styles);

const INIT_FORM: OptionalRecord<CarServiceFormSchema> = {
    car: undefined,

    since: undefined,
    since_time: DEFAULT_SERVICE_TIME,
    until: undefined,
    until_time: DEFAULT_SERVICE_TIME,

    job_type: undefined,
    mileage: '',
    job_comment: '',
};

const VALIDATION_RULES: UseFormValidation<OptionalRecord<CarServiceFormSchema>> = {
    car: validateRequired(i18n('Select car')),

    since: validateRequired(i18n('Enter date and time')), // todo validate date
    since_time: validateRequired(i18n('Enter date and time')), // todo validate time

    mileage: validateNumber({ min: 0 }),
};

export const CarServiceForm: React.FC<CarServiceFormProps> = function CarServiceForm({
    className,
    initial,
    controller,
    disableCarSwitch,
    onFormChange,
}) {
    const { update, watch, getValue, setValue, errors } = useForm<OptionalRecord<CarServiceFormSchema>>({
        init: { ...INIT_FORM, ...initial },
        validation: VALIDATION_RULES,
        controller,
        onFormChange,
    });

    const since = watch('since');
    const until = watch('until');
    const untilTime = watch('until_time');
    const isTypeMaintenance = watch('job_type') === CarServiceJobType.SCHEDULED_MAINTENANCE;

    React.useEffect(() => {
        setValue('mileage', isTypeMaintenance ? initial?.mileage || '' : '');

        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [isTypeMaintenance]);

    const onSetCurrentDateHandler = React.useCallback(() => {
        const now = new Date();

        setValue('until', now);
        setValue('until_time', formatTime(now));
    }, []);

    const today = React.useMemo(() => getDateStartDay(new Date()), []);

    const showTooltip = disableCarSwitch && !initial?.until && since && since < today && !until && untilTime !== '';

    return (
        <FormContainer className={className}>
            {!disableCarSwitch && (
                <FormLabel title={i18n('Car')}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.car} />}>
                        <CarInput
                            inputSize={InputSize.M}
                            placeholder={i18n('Car model or licence plate number')}
                            initialValue={getValue('car')}
                            onCarInputChange={update('car')}
                            hasError={Boolean(errors.car)}
                        />
                    </FormErrorContainer>
                </FormLabel>
            )}

            <FormDateTimeRange<CarServiceFormSchema>
                title={i18n('Dates')}
                minBookingDate={today}
                update={update}
                watch={watch}
                errors={errors}
            />

            {showTooltip && (
                <Tooltip
                    className={styles.dateTooltip}
                    text={
                        <>
                            {i18n('The car is still in service now.')}

                            <Link
                                className={styles.tooltipLink}
                                onClick={onSetCurrentDateHandler}
                            >
                                {i18n('Move from service')}
                            </Link>
                        </>
                    }
                    showTail
                    tailSide={TooltipTailSide.TOP}
                    tailPosition={TooltipTailPosition.END}
                />
            )}

            <FormLabel title={i18n('Type of service')}>
                <Select
                    placeholder={i18n('Select')}
                    items={CAR_SERVICE_MENU_ITEMS}
                    checkedMenu={watch('job_type')}
                    onSelectChange={update('job_type')}
                />
            </FormLabel>

            {isTypeMaintenance && (
                <div className={cx(styles.item, { half: true })}>
                    <FormLabel title={i18n('Mileage, km')}>
                        <Input
                            inputSize={InputSize.M}
                            inputMode="numeric"
                            type="number"
                            value={watch('mileage')}
                            hasClear
                            onInputChange={update('mileage')}
                        />
                    </FormLabel>
                </div>
            )}

            <FormLabel title={i18n('Comment')}>
                <Textarea
                    textareaSize={TextareaSize.M}
                    expandable
                    value={watch('job_comment')}
                    hasClear
                    onTextareaChange={update('job_comment')}
                />
            </FormLabel>

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
