import * as React from 'react';
import { useHistory, useParams } from 'react-router-dom';

import { SignalsTableWidget } from 'widgets/SignalsTableWidget';

import { UseCarSignalsResource } from 'entities/Car/api/useCarSignals/useCarSignals';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Path } from 'shared/consts/Path';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { Button } from 'shared/ui/Button/Button';
import { Island } from 'shared/ui/Island/Island';

import { i18n } from 'features/CarSignals/ui/CarSignals/CarSignals.i18n';

import styles from 'features/CarSignals/ui/CarSignals/CarSignals.css';

export interface CarSignalsProps {
    resource: UseCarSignalsResource;
}

const MAX_TABLE_ROWS_LENGTH = 5;

export const CarSignals: React.FC<CarSignalsProps> = React.memo(function CarSignals({ resource }) {
    const history = useHistory();
    const params = useParams();
    const signalsData = useResourceReader(resource);
    const { signals, can_get_more_pages } = signalsData || {};

    const data = signals && signals.slice(0, MAX_TABLE_ROWS_LENGTH);
    const hasSignals = data && data.length > 0;

    const onMoreClick = React.useCallback(() => {
        history.push(generateRouterPath(Path.CAR_SIGNALS, params));
    }, [history, params]);

    return (
        <Island
            title={i18n('Signals')}
            className={styles.container}
        >
            {hasSignals ? (
                <div className={styles.tableContainer}>
                    <SignalsTableWidget
                        data={data}
                        isShortView
                    />
                </div>
            ) : (
                <div className={styles.noData}>{i18n('There is no data')}</div>
            )}

            {can_get_more_pages && (
                <Button
                    className={styles.button}
                    color={ButtonColor.SECONDARY}
                    size={ButtonSize.M}
                    onClick={onMoreClick}
                    label={i18n('View all signals')}
                />
            )}
        </Island>
    );
});
