import * as React from 'react';
import { useHistory, useLocation, useRouteMatch } from 'react-router-dom';
import cn from 'classnames';

import { isCarsGroupsAccess } from 'utils/isCarsGroupsAccess';
import { showCarsSchedule } from 'utils/showCarsSchedule';

import { CarsView } from 'entities/Car/consts/CarsView';
import { CARS_VIEW_CGI } from 'entities/Car/consts/constants';

import { Path } from 'shared/consts/Path';
import { TabSize } from 'shared/consts/TabSize';
import { TabItemOptions } from 'shared/ui/TabItem/TabItem';
import { Tabs } from 'shared/ui/Tabs/Tabs';

import { i18n } from 'features/CarsHeaderTabs/ui/CarsHeaderTabs/CarsHeaderTabs.i18n';

import styles from 'features/CarsHeaderTabs/ui/CarsHeaderTabs/CarsHeaderTabs.css';

enum CarsHeaderTabsValue {
    TABLE = 'table',
    MAP = 'map',
    GROUPS = 'groups',
    SCHEDULE = 'schedule',
}

export interface CarsHeaderTabsProps {
    className?: string;

    onChangeCarsView?(carsView: CarsView | null): void;
}

export const CarsHeaderTabs: React.FC<CarsHeaderTabsProps> = function CarsHeaderTabs({ className, onChangeCarsView }) {
    // @todo: move to page
    const location = useLocation();
    const history = useHistory();
    const isCarsGroupsRoute = useRouteMatch(Path.CARS_GROUPS);
    const isCarsScheduleRoute = useRouteMatch(Path.CARS_SCHEDULE);

    const [currentItem, setCurrentItem] = React.useState<string>();

    React.useEffect(() => {
        if (isCarsGroupsRoute || isCarsScheduleRoute) {
            return setCurrentItem(isCarsGroupsRoute ? CarsHeaderTabsValue.GROUPS : CarsHeaderTabsValue.SCHEDULE);
        }

        const searchParams = new URLSearchParams(location?.search);
        const carsViewParams = searchParams.get(CARS_VIEW_CGI) as CarsView;
        const carsView = [CarsView.TABLE, CarsView.MAP].includes(carsViewParams) ? carsViewParams : CarsView.TABLE;

        if (!carsViewParams) {
            searchParams.set(CARS_VIEW_CGI, carsView);
            history.push(`${Path.CARS}?${searchParams}`);
        }

        setCurrentItem(carsView);

        if (onChangeCarsView) {
            onChangeCarsView(carsView);
        }
    }, [location, isCarsGroupsRoute, isCarsScheduleRoute]);

    const onItemClickHandler = React.useCallback(
        (value: CarsHeaderTabsValue) => {
            const searchParams = new URLSearchParams(location?.search);

            setCurrentItem(value);

            switch (value) {
                case CarsHeaderTabsValue.TABLE:
                case CarsHeaderTabsValue.MAP:
                    const carsViewParams = searchParams.get(CARS_VIEW_CGI);

                    if (carsViewParams) {
                        searchParams.set(CARS_VIEW_CGI, value);
                        history.push(`${Path.CARS}?${searchParams}`);
                    } else {
                        const emptySearchParams = new URLSearchParams();

                        emptySearchParams.set(CARS_VIEW_CGI, value);
                        history.push(`${Path.CARS}?${emptySearchParams}`);
                    }

                    break;

                case CarsHeaderTabsValue.GROUPS:
                    history.push(Path.CARS_GROUPS);
                    break;

                case CarsHeaderTabsValue.SCHEDULE:
                    history.push(Path.CARS_SCHEDULE);
                    break;
            }
        },
        [location],
    );

    const hasCarsGroupsAccess = isCarsGroupsAccess();
    const hasCarsSchedule = showCarsSchedule();

    const items = React.useMemo(() => {
        let list: TabItemOptions[] = [
            { label: i18n('Table'), value: CarsHeaderTabsValue.TABLE },
            { label: i18n('Map'), value: CarsHeaderTabsValue.MAP },
        ];

        if (hasCarsGroupsAccess) {
            list.push({
                label: i18n('Groups'),
                value: CarsHeaderTabsValue.GROUPS,
            });
        }

        if (hasCarsSchedule) {
            list.push({
                label: i18n('Schedule'),
                value: CarsHeaderTabsValue.SCHEDULE,
            });
        }

        return list;
    }, [hasCarsGroupsAccess, hasCarsSchedule]);

    return (
        <Tabs
            className={cn(styles.carsHeaderTabs, className)}
            size={TabSize.M}
            items={items}
            currentItem={currentItem}
            onItemClick={onItemClickHandler}
        />
    );
};
