import * as React from 'react';

import { ONE_DAY } from 'constants/constants';

import { GRID_CELL_WIDTH } from 'features/CarsScheduleGrid/consts/constants';
import {
    CarsScheduleGridBody,
    CarsScheduleGridBodyProps,
} from 'features/CarsScheduleGrid/ui/CarsScheduleGridBody/CarsScheduleGridBody';
import { CarsScheduleGridHeader } from 'features/CarsScheduleGrid/ui/CarsScheduleGridHeader/CarsScheduleGridHeader';

import { getCarsScheduleGridMenuItems } from 'entities/Car';
import { CarsScheduleFormType } from 'entities/Car/consts/CarsScheduleFormType';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getDatesInYear } from 'shared/helpers/getDatesInYear/getDatesInYear';
import { useScrollSync } from 'shared/hooks/useScrollSync/useScrollSync';
import { Dropdown } from 'shared/ui/Dropdown/Dropdown';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';

import { i18n } from 'features/CarsScheduleGrid/ui/CarsScheduleGrid/CarsScheduleGrid.i18n';

import PlusIcon from 'shared/ui/Icons/images/plus-outline-16.inline.svg';

export interface CarsScheduleGridProps extends Omit<CarsScheduleGridBodyProps, 'months'> {
    className?: string;
}

const PREV_VISIBLE_DAYS = 3;

export const CarsScheduleGrid: React.FC<CarsScheduleGridProps> = function CarsScheduleGrid({
    className,
    items,
    hoveredItem,
    since,
    until,
    onCreate,
    onItemClick,
    onMouseEnterItem,
    onMouseLeaveItem,
}) {
    const calendarRef = React.useRef<Nullable<HTMLDivElement>>(null);
    const bodyRef = React.useRef<Nullable<HTMLDivElement>>(null);

    const goToDate = React.useCallback(
        (date: Date) => {
            const diff = date.getTime() - since.getTime();
            const days = Math.floor(diff / ONE_DAY);

            if (calendarRef.current) {
                calendarRef.current.scrollLeft = GRID_CELL_WIDTH * days;
            }
        },
        [since],
    );

    useScrollSync(calendarRef, bodyRef);

    React.useEffect(() => {
        goToDate(new Date(Date.now() - PREV_VISIBLE_DAYS * ONE_DAY));
    }, [goToDate]);

    const menuItems: MenuItemOptions<CarsScheduleFormType>[] = React.useMemo(
        () => getCarsScheduleGridMenuItems(onCreate),
        [onCreate],
    );

    const months = getDatesInYear(since, until);

    return (
        <div className={className}>
            <CarsScheduleGridHeader
                months={months}
                controls={
                    <Dropdown
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        icon={PlusIcon}
                        label={i18n('New')}
                        items={menuItems}
                        position="bottom-end"
                        hasArrow={false}
                    />
                }
                calendarRef={calendarRef}
            />

            <CarsScheduleGridBody
                items={items}
                hoveredItem={hoveredItem}
                since={since}
                until={until}
                months={months}
                onCreate={onCreate}
                onItemClick={onItemClick}
                onMouseEnterItem={onMouseEnterItem}
                onMouseLeaveItem={onMouseLeaveItem}
                ref={bodyRef}
            />
        </div>
    );
};
