import * as React from 'react';
import { limitShift, shift } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { ONE_DAY } from 'constants/constants';

import { GRID_CELL_WIDTH } from 'features/CarsScheduleGrid/consts/constants';
import { CarsScheduleGridCreateCallback } from 'features/CarsScheduleGrid/types/CarsScheduleGridCreateCallback';
import { CarsScheduleGridEditCallback } from 'features/CarsScheduleGrid/types/CarsScheduleGridEditCallback';
import { CarsScheduleGridCalendarProps } from 'features/CarsScheduleGrid/ui/CarsScheduleGridCalendar/CarsScheduleGridCalendar';
import { CarsScheduleGridItem } from 'features/CarsScheduleGrid/ui/CarsScheduleGridItem/CarsScheduleGridItem';

import { getCarsScheduleGridMenuItems } from 'entities/Car';
import { CarsScheduleFormType } from 'entities/Car/consts/CarsScheduleFormType';
import { CarOfferPreviewSchema } from 'entities/Car/types/CarOfferPreviewSchema';
import { CarsScheduleListItems } from 'entities/Car/ui/CarsScheduleList/CarsScheduleList';

import { getDateNextDay } from 'shared/helpers/getDateNextDay/getDateNextDay';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { useFloatingMenu } from 'shared/hooks/useFloatingMenu/useFloatingMenu';
import { Menu } from 'shared/ui/Menu/Menu';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';

import styles from 'features/CarsScheduleGrid/ui/CarsScheduleGridBody/CarsScheduleGridBody.css';

export interface CarsScheduleGridBodyProps extends Pick<CarsScheduleGridCalendarProps, 'months'> {
    className?: string;
    items: CarsScheduleListItems;
    hoveredItem?: string;
    since: Date;
    until: Date;

    onCreate?: CarsScheduleGridCreateCallback;
    onItemClick?: CarsScheduleGridEditCallback<CarOfferPreviewSchema>;

    onMouseEnterItem?(event: React.MouseEvent<HTMLLIElement>): void;

    onMouseLeaveItem?(): void;
}

const cx = cn.bind(styles);

const DAY_WIDTH = GRID_CELL_WIDTH;
const DEFAULT_SCHEDULE_HOURS = 9;

const INITIAL_MOUSE_STATE = {
    id: '',
    since: null,
    until: null,
};

function scheduleTime(date: Date, hours: number = DEFAULT_SCHEDULE_HOURS, minutes: number = 0): Date {
    const copy = new Date(date);

    copy.setHours(hours, minutes, 0, 0);

    return copy;
}

export const CarsScheduleGridBody = React.forwardRef<HTMLDivElement, CarsScheduleGridBodyProps>(
    function CarsScheduleGridBody(
        {
            className,
            months,
            items,
            hoveredItem,
            since,
            until,
            onCreate,
            onItemClick,
            onMouseEnterItem,
            onMouseLeaveItem,
        },
        ref,
    ) {
        const referenceRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;
        const floatingRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

        const { cars } = items;

        const time2pos = React.useCallback(
            (current: Date) => {
                return Math.round(((current.getTime() - since.getTime()) / ONE_DAY) * DAY_WIDTH);
            },
            [since],
        );

        const pos2time = React.useCallback(
            (pos: number) => {
                return new Date(since.getTime() + Math.round((pos / DAY_WIDTH) * ONE_DAY));
            },
            [since],
        );

        const width = time2pos(until);
        const currentTimeLeft = time2pos(new Date());

        const [mouseState, setMouseState] = React.useState<{
            id: string;
            since: Nullable<Date>;
            until: Nullable<Date>;
        }>(INITIAL_MOUSE_STATE);

        const onMenuCloseHandler = React.useCallback(() => {
            setMouseState(INITIAL_MOUSE_STATE);
        }, []);

        const onMouseUp = React.useCallback(() => {
            if (!mouseState.id) {
                return;
            }

            setMouseState({
                id: mouseState.id,
                since: mouseState.since,
                until: mouseState.until,
            });
        }, [mouseState]);

        const { style, isMenuClosing, isMenuVisible, onReferenceActionHandler, onMenuItemClickHandler } =
            useFloatingMenu<HTMLLIElement>({
                referenceRef,
                floatingRef,
                disabled: !mouseState.id,
                options: {
                    placement: 'bottom-end',
                    middleware: [
                        shift({
                            crossAxis: true,
                            padding: 8,
                            limiter: limitShift(),
                        }),
                    ],
                },
                onReferenceClick: onMouseUp,
                onMenuClose: onMenuCloseHandler,
            });

        const onMouseDown = React.useCallback(
            (event: React.MouseEvent) => {
                if (isMenuVisible) {
                    return;
                }

                const target = event.target as HTMLElement;
                const id = target.getAttribute && target.getAttribute('data-id');

                if (id) {
                    const time = pos2time(event.nativeEvent.offsetX);

                    setMouseState({
                        id,
                        since: scheduleTime(getDateStartDay(time)),
                        until: scheduleTime(getDateNextDay(time)),
                    });
                }
            },
            [pos2time, isMenuVisible],
        );

        const onMouseMove = React.useCallback(
            (event: React.MouseEvent) => {
                if (!mouseState.id || isMenuVisible) {
                    return;
                }

                const nextUntil = scheduleTime(getDateNextDay(pos2time(event.nativeEvent.offsetX)));

                if (nextUntil.getTime() !== mouseState.until?.getTime()) {
                    setMouseState({
                        id: mouseState.id,
                        since: mouseState.since,
                        until: nextUntil,
                    });
                }
            },
            [mouseState, pos2time, isMenuVisible],
        );

        const menuItems: MenuItemOptions<CarsScheduleFormType>[] = React.useMemo(() => {
            if (!onCreate) {
                return [];
            }

            return getCarsScheduleGridMenuItems(onCreate, {
                id: mouseState.id,
                since: mouseState.since!,
                until: mouseState.until!,
            });
        }, [onCreate, mouseState]);

        return (
            <div
                className={cn(styles.body)}
                ref={ref}
            >
                <div
                    className={styles.content}
                    style={{ width: width + 'px' }}
                >
                    <div className={styles.lines}>
                        {months.map(({ dates }, index) => (
                            <div
                                className={styles.lineMonth}
                                style={
                                    {
                                        '--month-title-width': `${dates.length * DAY_WIDTH}px`,
                                    } as React.CSSProperties
                                }
                                key={index}
                            />
                        ))}
                    </div>

                    <div
                        className={styles.currentTime}
                        style={{ left: currentTimeLeft + 'px' }}
                    />

                    <ul>
                        {cars?.map(({ id, car_details, offers }) => (
                            <li
                                className={cx(styles.item, { hovered: id === hoveredItem })}
                                onMouseEnter={onMouseEnterItem}
                                onMouseLeave={onMouseLeaveItem}
                                onMouseDown={onMouseDown}
                                onMouseUp={onReferenceActionHandler}
                                onMouseMove={onMouseMove}
                                data-id={id}
                                key={id}
                            >
                                {id === mouseState.id && (
                                    <>
                                        <CarsScheduleGridItem<unknown>
                                            className={cx(styles.selection, { current: true, hide: isMenuClosing })}
                                            gridStartDate={since}
                                            payload={{}}
                                            status="draft"
                                            since={mouseState.since!}
                                            until={mouseState.until!}
                                            setRef={referenceRef}
                                        />

                                        {isMenuVisible && menuItems && (
                                            <Menu
                                                className={cx(styles.menu, { close: isMenuClosing })}
                                                shadow
                                                style={style}
                                                items={menuItems}
                                                onItemClick={onMenuItemClickHandler}
                                                ref={floatingRef}
                                            />
                                        )}
                                    </>
                                )}

                                {offers?.map((offer) => (
                                    <CarsScheduleGridItem<CarOfferPreviewSchema>
                                        {...offer}
                                        className={cx(styles.selection, { disabled: mouseState.id })}
                                        gridStartDate={since}
                                        payload={{ car: car_details, ...offer }}
                                        since={offer.actual_since || offer.since}
                                        until={offer.actual_until || offer.until}
                                        key={offer.offer_id}
                                        onClick={onItemClick}
                                    />
                                ))}
                            </li>
                        ))}
                    </ul>
                </div>
            </div>
        );
    },
);
