import * as React from 'react';
import cn from 'classnames/bind';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { isDateWeekend } from 'shared/helpers/isDateWeekend/isDateWeekend';
import { isSameDate } from 'shared/helpers/isSameDate/isSameDate';

import styles from 'features/CarsScheduleGrid/ui/CarsScheduleGridCalendar/CarsScheduleGridCalendar.css';

export interface CarsScheduleGridCalendarProps {
    className?: string;
    months: { year: number; dates: Date[] }[];
}

const cx = cn.bind(styles);

export const CarsScheduleGridCalendar = React.forwardRef<HTMLDivElement, CarsScheduleGridCalendarProps>(
    function CarsScheduleGridCalendar({ className, months }, ref) {
        const monthTitleRef = React.useRef<Nullable<HTMLHeadingElement>>(null);

        const [monthTitleWidth, setMonthTitleWidth] = React.useState<number>();

        React.useEffect(() => {
            if (monthTitleRef && monthTitleRef.current) {
                // TODO: why???
                setMonthTitleWidth(monthTitleRef.current.offsetWidth);
            }
        }, [monthTitleRef]);

        const locale = getI18nLocale();
        const now = new Date();

        return (
            <div
                className={cn(styles.calendar, className)}
                ref={ref}
            >
                {months.map(({ year, dates }, index) => (
                    <div
                        className={styles.month}
                        key={index}
                    >
                        <h5
                            className={styles.monthTitle}
                            style={{ '--month-title-width': monthTitleWidth } as React.CSSProperties}
                            ref={monthTitleRef}
                        >
                            {getDateString(new Date(dates[0]), locale, { month: 'long' })}

                            <span className={styles.year}>{year}</span>
                        </h5>

                        <ul className={styles.days}>
                            {dates.map((date) => {
                                const isCurrentDay = isSameDate(now, date);
                                const isWeekend = isDateWeekend(date);

                                return (
                                    <li
                                        className={styles.day}
                                        key={date.getTime()}
                                    >
                                        <span
                                            className={cx(styles.date, {
                                                current: isCurrentDay,
                                                weekend: isWeekend,
                                            })}
                                        >
                                            {date.getDate()}
                                        </span>
                                    </li>
                                );
                            })}
                        </ul>
                    </div>
                ))}
            </div>
        );
    },
);
