import * as React from 'react';
import cn from 'classnames/bind';

import { ONE_DAY } from 'constants/constants';

import { GRID_CELL_WIDTH } from 'features/CarsScheduleGrid/consts/constants';
import { CarsScheduleGridEditCallback } from 'features/CarsScheduleGrid/types/CarsScheduleGridEditCallback';

import { CarOfferStatusSchema } from 'entities/Car/types/CarOfferStatusSchema';

import { useItemRef } from 'shared/hooks/useItemRef/useItemRef';

import styles from 'features/CarsScheduleGrid/ui/CarsScheduleGridItem/CarsScheduleGridItem.css';

export interface CarsScheduleGridItemProps<T> {
    className?: string;
    gridStartDate: Date;

    payload: T;
    status: CarOfferStatusSchema;
    since: Date;
    until: Date;

    onClick?: CarsScheduleGridEditCallback<T>;

    setRef?: ((element: Nullable<HTMLDivElement>) => void) | React.MutableRefObject<Nullable<HTMLDivElement>>;
}

const TEN = 10;
const DESC_MIN_WIDTH = 30;
const DESC_MID_WIDTH = 60;

const cx = cn.bind(styles);

function getTime(date: Date): React.ReactNode {
    return (
        <>
            {date.getHours()}

            <sup>{date.getMinutes() < TEN ? '0' + date.getMinutes() : String(date.getMinutes())}</sup>
        </>
    );
}

export function CarsScheduleGridItem<T>({
    className,
    gridStartDate,
    payload,
    status,
    since,
    until,
    onClick,
    setRef,
}: CarsScheduleGridItemProps<T>) {
    const { ref, itemRef } = useItemRef<HTMLDivElement>(setRef);

    const width = Math.round(((until.getTime() - since.getTime()) / ONE_DAY) * GRID_CELL_WIDTH);
    const left = Math.round(((since.getTime() - gridStartDate.getTime()) / ONE_DAY) * GRID_CELL_WIDTH);

    const onItemClick = React.useCallback(() => {
        if (onClick) {
            onClick(payload, ref);
        }
    }, [payload, onClick]);

    return (
        <div
            className={cx(styles.item, [status.toLowerCase(), className])}
            style={{ left: left + 'px', width: width + 'px' }}
            onClick={onItemClick}
            ref={itemRef}
        >
            {width >= DESC_MIN_WIDTH && (
                <>
                    <span className={styles.time}>{getTime(since)}</span>

                    {width >= DESC_MID_WIDTH && <span className={styles.time}>{getTime(until)}</span>}
                </>
            )}
        </div>
    );
}
