import * as React from 'react';
import { Placement } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { usePortal } from 'shared/hooks/usePortal/usePortal';
import { Button } from 'shared/ui/Button/Button';
import { Widget } from 'shared/ui/Widget/Widget';

import ArrowIcon from 'shared/ui/Icons/images/arrow-24x8.inline.svg';
import CloseIcon from 'shared/ui/Icons/images/close-16.inline.svg';

import styles from 'features/CarsSchedulePopup/ui/CarsScheduleBasePopup/CarsScheduleBasePopup.css';

export interface CarsScheduleBasePopupUIProps {
    style?: React.CSSProperties;
    placement?: Placement;
    isClosing?: boolean;

    tailRef?: React.Ref<HTMLSpanElement>;
    tailStyle?: React.CSSProperties;
    isTailHidden?: boolean;

    onClose?(): void;
}

export interface CarsScheduleBasePopupProps extends CarsScheduleBasePopupUIProps {
    className?: string;

    title: string | React.ReactNode;
    description?: string | React.ReactNode;
    controls?: React.ReactNode;

    children?: React.ReactNode;
}

const cx = cn.bind(styles);

export const CarsScheduleBasePopup = React.forwardRef<HTMLDivElement, CarsScheduleBasePopupProps>(
    function CarsScheduleBasePopup(
        {
            className,
            style,
            placement,
            isClosing,
            tailRef,
            tailStyle,
            isTailHidden,
            title,
            description,
            controls,
            onClose,
            children,
        },
        ref,
    ) {
        const Portal = usePortal();

        return (
            <Portal>
                <Widget
                    className={cx(styles.popup, { close: isClosing }, [placement])}
                    style={style}
                    role="dialog"
                    ref={ref}
                >
                    <span
                        className={cx(styles.tail, { hidden: isTailHidden })}
                        style={tailStyle}
                        ref={tailRef}
                    >
                        <ArrowIcon className={styles.tailIcon} />
                    </span>

                    <Button
                        className={styles.close}
                        size={ButtonSize.M}
                        color={ButtonColor.CLEAR}
                        icon={CloseIcon}
                        round
                        onClick={onClose}
                    />

                    {title && (
                        <div className={styles.header}>
                            <h3 className={styles.title}>{title}</h3>

                            {typeof description === 'string' ? (
                                <p className={styles.description}>{description}</p>
                            ) : (
                                description && <div className={styles.description}>{description}</div>
                            )}
                        </div>
                    )}

                    {children}

                    {controls && <div className={styles.controls}>{controls}</div>}
                </Widget>
            </Portal>
        );
    },
);
