import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';
import cn from 'classnames/bind';

import { MODAL_BACK, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { isRentalRetireeFlowFlag } from 'utils/isRentalRetireeFlowFlag';

import { getCarsSchedulePopupTitle } from 'features/CarsSchedulePopup/helpers/getCarsSchedulePopupTitle/getCarsSchedulePopupTitle';
import {
    CarsScheduleBasePopup,
    CarsScheduleBasePopupUIProps,
} from 'features/CarsSchedulePopup/ui/CarsScheduleBasePopup/CarsScheduleBasePopup';
import { CarsSchedulePopupComment } from 'features/CarsSchedulePopup/ui/CarsSchedulePopupComment/CarsSchedulePopupComment';
import { CarsSchedulePopupDetail } from 'features/CarsSchedulePopup/ui/CarsSchedulePopupDetail/CarsSchedulePopupDetail';
import { CarsSchedulePopupInfo } from 'features/CarsSchedulePopup/ui/CarsSchedulePopupInfo/CarsSchedulePopupInfo';
import { CarsSchedulePopupUser } from 'features/CarsSchedulePopup/ui/CarsSchedulePopupUser/CarsSchedulePopupUser';

import { CarRentalRetireeStepSchema, getCarOfferCurrencyName } from 'entities/Car';
import { fetchCarRentalAgreement } from 'entities/Car/api/fetchCarRentalAgreement/fetchCarRentalAgreement';
import { useCarOffer } from 'entities/Car/api/useCarOffer/useCarOffer';
import { CarOfferSessionStatus } from 'entities/Car/consts/CarOfferSessionStatus';
import { getCarOfferSessionStatus } from 'entities/Car/helpers/getCarOfferSessionStatus/getCarOfferSessionStatus';
import { CarOfferConfigSchema } from 'entities/Car/types/CarOfferConfigSchema';
import { CarOfferSchema } from 'entities/Car/types/CarOfferSchema';
import { UserStatusInfo } from 'entities/User';
import { UserShortSchema } from 'entities/User/types/UserShortSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { LinkColor } from 'shared/consts/LinkColor';
import { TextWithDotSize } from 'shared/consts/TextWithDotSize';
import { downloadFile } from 'shared/helpers/downloadFile/downloadFile';
import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';
import { Button } from 'shared/ui/Button/Button';
import { Link } from 'shared/ui/Link/Link';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import { ModalObjectTypes } from 'components/types';

import { i18n } from 'features/CarsSchedulePopup/ui/CarsScheduleBookingPopup/CarsScheduleBookingPopup.i18n';

import EditIcon from 'shared/ui/Icons/images/edit-outline-16.inline.svg';
import PlayIcon from 'shared/ui/Icons/images/play-outline-16.inline.svg';
import StopIcon from 'shared/ui/Icons/images/stop-16.inline.svg';

import styles from 'features/CarsSchedulePopup/ui/CarsScheduleBookingPopup/CarsScheduleBookingPopup.css';

export interface CarsScheduleBookingPopupProps extends CarsScheduleBasePopupUIProps {
    className?: string;
    offerId: string;
    user?: UserShortSchema;

    onEdit?(): void;

    onStart?(step: CarRentalRetireeStepSchema): void;

    onFinish?(): void;

    onClose?(): void;
}

const cx = cn.bind(styles);

function getCost(
    value: string | number,
    currency: CarOfferSchema['currency'],
    currencies: CarOfferConfigSchema['currencies'],
): string {
    return `${formatNumber(Number(value))} ${getCarOfferCurrencyName(currency, currencies)}`;
}

export const CarsScheduleBookingPopup = React.forwardRef<HTMLDivElement, CarsScheduleBookingPopupProps>(
    function CarsScheduleBookingPopup(
        { className, offerId, user, onEdit, onStart, onFinish, onClose, ...otherProps },
        ref,
    ) {
        const history = useHistory();
        const location = useLocation();

        const { data, isLoading } = useCarOffer(offerId);

        const onStartHandler = React.useCallback(() => {
            const stage = data?.session.stage;

            if (onStart) {
                onStart(stage === 'old_state_acceptance' ? 'start' : 'documents');
            }
        }, [onStart, data]);

        const onTitleClickHandler = React.useCallback(() => {
            let searchParams = new URLSearchParams(location.search);
            searchParams.set(MODAL_BACK, `${location.pathname}?${searchParams}`);

            let params = [
                {
                    key: MODAL_OBJECT_TYPE_CGI,
                    value: ModalObjectTypes.SESSION,
                },

                {
                    key: MODAL_OBJECT_ID_CGI,
                    value: offerId,
                },
            ];

            params.forEach(({ key, value }) => {
                searchParams.set(key, value);
            });

            history.push(`${location.pathname}?${searchParams}`);

            if (onClose) {
                onClose();
            }
        }, [offerId, onClose, location]);

        const onRentalDownloadHandler = React.useCallback(async () => {
            try {
                const file = await fetchCarRentalAgreement(offerId, user?.id!);

                downloadFile(file);

                // @todo: add notification
            } catch (error) {
                console.error(error);

                // @todo: add notification
            }
        }, [offerId, user]);

        if (!data || isLoading) {
            return null;
        }

        const { status, delivery_location, return_location, currency, deposit, total_payment, comment } = data.offer;
        const { stage, since, until, actual_since, actual_until } = data.session;

        const sessionStatus = getCarOfferSessionStatus(data.session);
        const isStatusInProgress =
            stage !== 'old_state_acceptance' && sessionStatus === CarOfferSessionStatus.IN_PROGRESS;
        const isCompletedStatus =
            sessionStatus === CarOfferSessionStatus.COMPLETED || sessionStatus === CarOfferSessionStatus.OUTDATED;

        const userStatus = user?.status;

        return (
            <CarsScheduleBasePopup
                {...otherProps}
                className={className}
                title={
                    <TextWithDot
                        className={cx(styles.dot, [status.toLowerCase()])}
                        size={TextWithDotSize.L}
                    >
                        <Link
                            color={LinkColor.BLACK}
                            onClick={onTitleClickHandler}
                        >
                            {getCarsSchedulePopupTitle(status, actual_since, actual_until)}
                        </Link>
                    </TextWithDot>
                }
                description={<Link onClick={onRentalDownloadHandler}>{i18n('Download agreement')}</Link>}
                controls={
                    <>
                        <Button
                            size={ButtonSize.M}
                            color={ButtonColor.SECONDARY}
                            icon={!isCompletedStatus ? EditIcon : undefined}
                            label={isCompletedStatus ? i18n('View booking') : i18n('Edit')}
                            wide
                            onClick={onEdit}
                        />

                        {isRentalRetireeFlowFlag() && !isCompletedStatus && (
                            <Button
                                size={ButtonSize.M}
                                color={ButtonColor.SECONDARY}
                                icon={isStatusInProgress ? StopIcon : PlayIcon}
                                label={isStatusInProgress ? i18n('Finish ride') : i18n('Start ride')}
                                wide
                                onClick={isStatusInProgress ? onFinish : onStartHandler}
                            />
                        )}
                    </>
                }
                onClose={onClose}
                ref={ref}
            >
                <div className={styles.content}>
                    <CarsSchedulePopupInfo date={actual_since ?? since}>
                        <p
                            className={styles.location}
                            title={delivery_location.name}
                        >
                            {delivery_location.name}
                        </p>
                    </CarsSchedulePopupInfo>

                    <CarsSchedulePopupInfo date={actual_until ?? until}>
                        <p
                            className={styles.location}
                            title={return_location.name}
                        >
                            {return_location.name}
                        </p>
                    </CarsSchedulePopupInfo>

                    {user && (
                        <CarsSchedulePopupDetail
                            label={
                                <UserStatusInfo
                                    className={styles.info}
                                    status={userStatus}
                                    withoutIcon
                                />
                            }
                        >
                            <CarsSchedulePopupUser user={user} />
                        </CarsSchedulePopupDetail>
                    )}

                    <div className={styles.item}>
                        <CarsSchedulePopupDetail label={i18n('Total amount')}>
                            {getCost(total_payment, currency, data.offerConfig.currencies)}
                        </CarsSchedulePopupDetail>

                        <CarsSchedulePopupDetail label={i18n('Deposit')}>
                            {getCost(deposit, currency, data.offerConfig.currencies)}
                        </CarsSchedulePopupDetail>
                    </div>

                    <CarsSchedulePopupComment comment={comment} />
                </div>
            </CarsScheduleBasePopup>
        );
    },
);
