import * as React from 'react';
import { useState } from 'react';
import { arrow, flip, shift } from '@floating-ui/react-dom';

import {
    CarsScheduleBookingPopup,
    CarsScheduleBookingPopupProps,
} from 'features/CarsSchedulePopup/ui/CarsScheduleBookingPopup/CarsScheduleBookingPopup';
import { CarsScheduleServicePopup } from 'features/CarsSchedulePopup/ui/CarsScheduleServicePopup/CarsScheduleServicePopup';

import { CarRentalRetireeStepSchema } from 'entities/Car';
import { CarsScheduleFormType } from 'entities/Car/consts/CarsScheduleFormType';
import { CarOfferStatusSchema } from 'entities/Car/types/CarOfferStatusSchema';
import { UserShortSchema } from 'entities/User/types/UserShortSchema';

import { useClickOutside } from 'shared/hooks/useClickOutside/useClickOutside';
import { useFloatingUI } from 'shared/hooks/useFloatingUI/useFloatingUI';
import { useTimerRef } from 'shared/hooks/useTimerRef/useTimerRef';

export interface CarsSchedulePopupProps extends Pick<CarsScheduleBookingPopupProps, 'onStart' | 'onFinish'> {
    offerId: string;
    tagId: string;
    user?: UserShortSchema;
    status: CarOfferStatusSchema;

    onEdit?(type: CarsScheduleFormType): void;

    onClose?(): void;

    referenceRef: React.MutableRefObject<Nullable<HTMLElement>>;
}

const CLOSE_ANIMATION_TIME = 150;
const OFFSET = 16;
const SHIFT_PADDING = { top: 144, right: 16, bottom: 0, left: 400 };
const TAIL_PADDING = 24;

export const CarsSchedulePopup: React.FC<CarsSchedulePopupProps> = function CarsSchedulePopup({
    user,
    offerId,
    tagId,
    status,
    onEdit,
    onStart,
    onFinish,
    onClose,
    referenceRef,
}) {
    const timerRef = useTimerRef(0);
    const floatingRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;
    const tailRef = React.useRef() as React.MutableRefObject<Nullable<HTMLSpanElement>>;

    const [, setVer] = useState<number>(0);
    const [isClosing, setClosing] = React.useState<boolean>(false);
    const [isVisible, setVisible] = React.useState<boolean>(true);

    const contentType = status === 'service' ? CarsScheduleFormType.SERVICE : CarsScheduleFormType.BOOKING;

    const { placement, middlewareData, style } = useFloatingUI(referenceRef, floatingRef, isVisible, {
        placement: 'top',
        middleware: [
            flip({ crossAxis: false }),
            shift({
                crossAxis: true,
                padding: SHIFT_PADDING,
            }),
            arrow({
                element: tailRef,
                padding: TAIL_PADDING,
            }),
        ],
        offset: OFFSET,
    });

    const { arrow: tail, shift: popupShift } = middlewareData;

    const tailStyle: React.CSSProperties = {
        top: tail?.y ?? '',
        left: tail?.x ?? '',
    };
    const isTailHidden = Boolean(tail?.centerOffset !== 0 || popupShift?.y);

    const setFloatingRef = React.useCallback((ref: Nullable<HTMLDivElement>) => {
        if (floatingRef.current !== ref) {
            floatingRef.current = ref;

            setVer((ver) => ver + 1);
        }
    }, []);

    const onCloseHandler = React.useCallback(() => {
        if (!isClosing) {
            setClosing(true);

            timerRef.current = setTimeout(() => {
                setClosing(false);
                setVisible(false);

                if (onClose) {
                    onClose();
                }
            }, CLOSE_ANIMATION_TIME) as unknown as number;
        }
    }, [onClose, isClosing]);

    const onOutsideClickHandler = React.useCallback(
        ({ target }: MouseEvent) => {
            if (!referenceRef.current?.contains(target as Node)) {
                onCloseHandler();
            }
        },
        [referenceRef.current, onCloseHandler],
    );

    useClickOutside(floatingRef, onOutsideClickHandler);

    const onEditHandler = React.useCallback(() => {
        if (onEdit) {
            onEdit(contentType);
        }

        onCloseHandler();
    }, [onEdit, contentType, onCloseHandler]);

    const onStartHandler = React.useCallback(
        (step: CarRentalRetireeStepSchema) => {
            if (onStart) {
                onStart(step);
            }

            onCloseHandler();
        },
        [onStart, onCloseHandler],
    );

    const onFinishHandler = React.useCallback(() => {
        if (onFinish) {
            onFinish();
        }

        onCloseHandler();
    }, [onFinish, onCloseHandler]);

    if (!isVisible) {
        return null;
    }

    const popupProps = {
        placement,
        style,
        isClosing,
        tailRef,
        tailStyle,
        isTailHidden,
        onEdit: onEditHandler,
        onClose: onCloseHandler,
        ref: setFloatingRef,
    };

    return (
        <>
            {contentType === CarsScheduleFormType.BOOKING && (
                <CarsScheduleBookingPopup
                    {...popupProps}
                    offerId={offerId}
                    user={user}
                    onStart={onStartHandler}
                    onFinish={onFinishHandler}
                />
            )}

            {contentType === CarsScheduleFormType.SERVICE && (
                <CarsScheduleServicePopup
                    {...popupProps}
                    tagId={tagId}
                />
            )}
        </>
    );
};
